
// SPDX-License-Identifier: CC-BY-NC-SA-4.0
//
// Copyright (C) 2025 Bit by Bit Signal Processing LLC  (https://bxbsp.com)
//
// This work is placed under the "Creative Commons Attribution
// NonCommercial ShareAlike 4.0 International" license, known
// by the shortened acronym "CC-BY-NC-SA-4.0".
//
// This work is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//
// A CC-BY-NC-SA-4.0 license allows you to use, distribute, and modify
// this work, so long as such uses are non-commercial in nature,
// so long as any derived works are offered on the same terms,
// and so long as attribution is given to the original author.
// For further details, see the Creative Commons License
// "CC-BY-NC-SA-4.0".
//
// You should have received a copy of the CC-BY-NC-SA-4.0 license
// along with this work. If not, see
// <https://creativecommons.org/licenses/by-nc-sa/4.0/>.
//



#include "data_sources.hh"
#include <stdlib.h>
#include <stdio.h>

extern bool RFSoC4x2;

void adc_sw_number_bounds_check(int adc_number_sw)
{
  if(RFSoC4x2 && adc_number_sw>=0 && adc_number_sw<4)
    return;
  else if(adc_number_sw>=0 && adc_number_sw<8)
    return;

  printf("Error:  adc_number_sw %d out of bounds in adc_sw_number_bounds_check!\n", adc_number_sw);
  abort();
}


void adc_hw_number_bounds_check(int adc_number_hw)
{
  if(RFSoC4x2 && adc_number_hw>=0 && adc_number_hw<4)
    return;
  else if(adc_number_hw>=0 && adc_number_hw<8)
    return;

  printf("Error:  adc_number_hw %d out of bounds in adc_hw_number_bounds_check!\n", adc_number_hw);
  abort();
}


void dac_sw_number_bounds_check(int dac_number_sw)
{
  if(RFSoC4x2 && dac_number_sw>=0 && dac_number_sw<2)
    return;
  else if(dac_number_sw>=0 && dac_number_sw<8)
    return;

  printf("Error:  dac_number_sw %d out of bounds in dac_sw_number_bounds_check!\n", dac_number_sw);
  abort();
}


void dac_hw_number_bounds_check(int dac_number_hw)
{
  if(RFSoC4x2 && dac_number_hw>=0 && dac_number_hw<2)
    return;
  else if(dac_number_hw>=0 && dac_number_hw<8)
    return;

  printf("Error:  dac_number_hw %d out of bounds in dac_hw_number_bounds_check!\n", dac_number_hw);
  abort();
}


int sw_adc_to_hw_adc(int adc_number_sw)
{
  adc_sw_number_bounds_check(adc_number_sw);

  if(RFSoC4x2)
    return 3-adc_number_sw;

  return adc_number_sw;
}

int hw_adc_to_sw_adc(int adc_number_hw)
{
  adc_hw_number_bounds_check(adc_number_hw);

  if(RFSoC4x2)
    return 3-adc_number_hw;

  return adc_number_hw;
}


int sw_dac_to_hw_dac(int dac_number_sw)
{
  dac_sw_number_bounds_check(dac_number_sw);

  if(RFSoC4x2)
    return 1-dac_number_sw;

  return dac_number_sw;
}


int hw_dac_to_sw_dac(int dac_number_hw)
{
  dac_hw_number_bounds_check(dac_number_hw);

  if(RFSoC4x2)
    return 1-dac_number_hw;

  return dac_number_hw;
}




const char* adc_names_sw  (int adc_number_sw)
{
  const static char* ADC_NAMES_RFSoC[] = { "ADC A", "ADC B", "ADC C", "ADC D" };
  const static char* ADC_NAMES[]       = { "ADC 0", "ADC 1", "ADC 2", "ADC 3", "ADC 4", "ADC 5", "ADC 6", "ADC 7" } ;

  adc_sw_number_bounds_check(adc_number_sw);

  if(RFSoC4x2)
    return ADC_NAMES_RFSoC[adc_number_sw];

  return ADC_NAMES[adc_number_sw];
}


const char* adc_names_hw  (int adc_number_hw)
{
  return adc_names_sw(hw_adc_to_sw_adc(adc_number_hw));
}


const char* dac_names_sw  (int dac_number_sw)
{
  const static char* DAC_NAMES_RFSoC[] = { "DAC A", "DAC B" };
  const static char* DAC_NAMES[]       = { "DAC 0", "DAC 1", "DAC 2", "DAC 3", "DAC 4", "DAC 5", "DAC 6", "DAC 7" } ;

  dac_sw_number_bounds_check(dac_number_sw);

  if(RFSoC4x2)
    return DAC_NAMES_RFSoC[dac_number_sw];
  
  return DAC_NAMES[dac_number_sw];
}


const char* dac_names_hw  (int dac_number_hw)
{
  return dac_names_sw(hw_dac_to_sw_dac(dac_number_hw));
}


color adc_colors_sw  (int adc_number_sw)
{
  const static color ADC_COLORS[] = { RED, GREEN, BLUE, ORANGE, BLUEGREEN, PURPLE, YELLOW, GREY6 };

  adc_sw_number_bounds_check(adc_number_sw);

  return ADC_COLORS[adc_number_sw];
}


color adc_colors_hw  (int adc_number_hw)
{
  return adc_colors_sw(hw_adc_to_sw_adc(adc_number_hw));
}


color dac_colors_sw  (int dac_number_sw)
{
  const static color DAC_COLORS[] = { RED, GREEN, BLUE, ORANGE, BLUEGREEN, PURPLE, YELLOW, GREY6 };

  dac_sw_number_bounds_check(dac_number_sw);

  return DAC_COLORS[dac_number_sw];
}


color dac_colors_hw  (int dac_number_hw)
{
  return dac_colors_sw(hw_dac_to_sw_dac(dac_number_hw));
}


const char* source_names_sw  (int source_number_sw)
{
  if(sw_source_is_dac(source_number_sw))
    return dac_names_sw(sw_source_to_sw_dac(source_number_sw));
  else
    return adc_names_sw(sw_source_to_sw_adc(source_number_sw));
}


color source_colors_sw (int source_number_sw)
{
  if(sw_source_is_dac(source_number_sw))
    return dac_colors_sw(sw_source_to_sw_dac(source_number_sw));
  else
    return adc_colors_sw(sw_source_to_sw_adc(source_number_sw));
}

int sw_source_to_hw_source(int sw_source)
{
  if(sw_source_is_dac(sw_source))
    return sw_dac_to_hw_dac(sw_source_to_sw_dac(sw_source)) + 8;
  else
    return sw_adc_to_hw_adc(sw_source_to_sw_adc(sw_source));
  
}

bool sw_source_is_adc(int sw_source)
{
  if(sw_source>=8)
    return false;

  adc_sw_number_bounds_check(sw_source);

  return true;
}

bool sw_source_is_dac(int sw_source)
{
  if(sw_source<8)
    return false;

  int dac_number_sw = sw_source - 8;
  
  dac_sw_number_bounds_check(dac_number_sw);

  return true;
}
  
int  sw_source_to_sw_adc(int sw_source)
{
  adc_sw_number_bounds_check(sw_source);

  return sw_source;
}

int  sw_source_to_sw_dac(int sw_source)
{
  int dac_number_sw = sw_source - 8;
  
  dac_sw_number_bounds_check(dac_number_sw);

  return dac_number_sw;
}


int  sw_source_to_hw_adc(int sw_source)
{
  int sw_adc = sw_source_to_sw_adc(sw_source);
  return sw_adc_to_hw_adc(sw_adc);
}

int  sw_source_to_hw_dac(int sw_source)
{
  int sw_dac = sw_source_to_sw_dac(sw_source);
  return sw_dac_to_hw_dac(sw_dac);
}
