
// SPDX-License-Identifier: CC-BY-NC-SA-4.0
//
// Copyright (C) 2025 Bit by Bit Signal Processing LLC  (https://bxbsp.com)
//
// This work is placed under the "Creative Commons Attribution
// NonCommercial ShareAlike 4.0 International" license, known
// by the shortened acronym "CC-BY-NC-SA-4.0".
//
// This work is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//
// A CC-BY-NC-SA-4.0 license allows you to use, distribute, and modify
// this work, so long as such uses are non-commercial in nature,
// so long as any derived works are offered on the same terms,
// and so long as attribution is given to the original author.
// For further details, see the Creative Commons License
// "CC-BY-NC-SA-4.0".
//
// You should have received a copy of the CC-BY-NC-SA-4.0 license
// along with this work. If not, see
// <https://creativecommons.org/licenses/by-nc-sa/4.0/>.
//


#include <string.h>

#include "menu_adc_control.hh"
#include "hwalloc.hh"
#include "data_update.hh"
#include "display.hh"

extern double sampling_rate_kHz;  // An integer, and thus perfectly accurate
extern double sampling_rate_MHz;  // Fractional
extern XRFdc *RFdcInstPtr;

int NUM_ADCS = 8;

static int limit(int v, int minval, int maxval)
{
  if(v<minval)
    return minval;
  if(v>maxval)
    return maxval;

  return v;
}

void menu_adc_control::update_settings_from_hw()
{
  int adc_number_hw = adc->adc_number_hw;
  //int adc_number_sw = hw_adc_to_sw_adc(adc_number_hw);

  u32 zone_u32;
  u32 dither_u32;
  u8  calibration_mode_u8;
  XRFdc_Cal_Freeze_Settings freeze_in;

  XRFdc_GetNyquistZone(RFdcInstPtr,  XRFDC_ADC_TILE, adcs[adc_number_hw].adc_tile, adcs[adc_number_hw].adc_block, &zone_u32);
  XRFdc_GetDither(RFdcInstPtr, adcs[adc_number_hw].adc_tile, adcs[adc_number_hw].adc_block, &dither_u32);
  XRFdc_GetCalFreeze(RFdcInstPtr, adcs[adc_number_hw].adc_tile, adcs[adc_number_hw].adc_block, &freeze_in);
  XRFdc_GetCalibrationMode(RFdcInstPtr, adcs[adc_number_hw].adc_tile, adcs[adc_number_hw].adc_block, &calibration_mode_u8);

  nyquist_button->select_no_event(zone_u32-1);                      // selection 0 is zone 1, selection 1 is zone 2
  dither_button->select_no_event(dither_u32);                       // selection 0 is no dither, selection 1 is dither
  calibration_freeze_button->select_no_event(freeze_in.CalFrozen);  // selection 0 is operating, selection 1 is frozen

  //printf("Freeze state is %d\n", freeze_in.CalFrozen);
  int mode = calibration_mode_u8 - ((XRFdc_Version >= XRFDC_GEN3) ? 0 : 1);


  // For Gen1 RFSoC: selection 0 is Mode_1, selection 1 is Mode_2
  // For Gen3 RFSoC: selectoin 0 is Auto, selection 1 is Mode_1, selection 2 is Mode_2
  // HW is always 0=AUTO, 1=MODE1, 2=MODE2
  calibration_mode_button->select_no_event(mode);
}

void menu_adc_control::apply_settings()
{
  int zone;
  u32 zone_u32;
  u32 dither_u32;
  u8  calibration_mode_u8;
  
  bool calibration_freeze  = (calibration_freeze_button->get_item_number_selected()==1);
  bool dither_enabled      = (dither_button->get_item_number_selected()==1);
  int  nyquist_num         = nyquist_button->get_item_number_selected();
  int  nyquist_value       = nyquist_num + 1;
  u8   calibration_num     = calibration_mode_button->get_item_number_selected();
  int  calibration_value   = calibration_num + (XRFdc_Version < XRFDC_GEN3 ? 1 : 0);
  
  int adc_number_hw = adc->adc_number_hw;
  int adc_number_sw = hw_adc_to_sw_adc(adc_number_hw);

  XRFdc_SetNyquistZone(RFdcInstPtr, XRFDC_ADC_TILE, adcs[adc_number_hw].adc_tile, adcs[adc_number_hw].adc_block, nyquist_value);
  XRFdc_GetNyquistZone(RFdcInstPtr,  XRFDC_ADC_TILE, adcs[adc_number_hw].adc_tile, adcs[adc_number_hw].adc_block, &zone_u32);

  zone = zone_u32;

  if(zone!=nyquist_value)
    {
      // Somehow, the zone we're in isn't the same as was requested
      printf("Nyquist zone change failed for %s. Was %d failed to change to %d now at %d.\n", adc_names_sw(adc_number_sw), adc->nyquist_zone, nyquist_value, zone);
      global_entry_rejected_beep();
    }

  if(zone!=adc->nyquist_zone)
    {
      //
      // The reset seems to mess even with the Nyquist zone.  (?)
      //
      // To be safe, reset ADC after Nyquist zone change.  Note that this can't be done
      // after calibration or dither change, because the reset alters them.
      //
      //int Status = XRFdc_Reset(RFdcInstPtr, XRFDC_ADC_TILE, adcs[adc_number].adc_tile);
      //
      //if(Status != XRFDC_SUCCESS)
      //{
      //  printf("WARNING:  ADC tile didn't fully reset after Nyquist zone change.\n");
      //}
      
      printf("Nyquist zone changed for %s from %d to %d.\n", adc_names_sw(adc_number_sw), adc->nyquist_zone, zone);
      adc->nyquist_zone = zone;
      //global_entry_accepted_beep();
    }

  uint32_t dither_requested = dither_enabled ? XRFDC_DITH_ENABLE : XRFDC_DITH_DISABLE;
  XRFdc_SetDither(RFdcInstPtr, adcs[adc_number_hw].adc_tile, adcs[adc_number_hw].adc_block, dither_requested);
  XRFdc_GetDither(RFdcInstPtr, adcs[adc_number_hw].adc_tile, adcs[adc_number_hw].adc_block, &dither_u32);

  bool dither_after_set = (dither_u32==XRFDC_DITH_ENABLE);

  if(dither_u32 != dither_requested)
    {
      // Somehow, dither is not as requested.
      printf("Dither change failed for %s. Was %s failed to change to %s, now at %s.\n", adc_names_sw(adc_number_sw), adc->dither_enabled ? "enabled" : "disabled", dither_requested ? "enabled" : "disabled", dither_after_set ? "enabled" : "disabled");
      global_entry_rejected_beep();
    }

  if(dither_after_set!=adc->dither_enabled)
    {
      printf("Dither changed for %s from %s to %s.\n", adc_names_sw(adc_number_sw), adc->dither_enabled ? "enabled" : "disabled", dither_after_set ? "enabled" : "disabled");
      adc->dither_enabled = dither_after_set;
      //global_entry_accepted_beep();
    }

  
  XRFdc_Cal_Freeze_Settings freeze_in;
  XRFdc_Cal_Freeze_Settings freeze_out;

  XRFdc_GetCalFreeze(RFdcInstPtr, adcs[adc_number_hw].adc_tile, adcs[adc_number_hw].adc_block, &freeze_in);
  freeze_in.CalFrozen          = calibration_freeze; /*Status indicates calibration freeze state*/
  freeze_in.DisableFreezePin   = true;               /*Disable the calibration freeze pin*/
  freeze_in.FreezeCalibration  = calibration_freeze; /*Setter for freezing*/
  XRFdc_SetCalFreeze(RFdcInstPtr, adcs[adc_number_hw].adc_tile, adcs[adc_number_hw].adc_block, &freeze_in);
  usleep(10);  // Freeze setting doesn't take effect immediately.
  freeze_out.CalFrozen          = !calibration_freeze; /*Status indicates calibration freeze state*/
  freeze_out.DisableFreezePin   = true;               /*Disable the calibration freeze pin*/
  freeze_out.FreezeCalibration  = !calibration_freeze; /*Setter for freezing*/
  XRFdc_GetCalFreeze(RFdcInstPtr, adcs[adc_number_hw].adc_tile, adcs[adc_number_hw].adc_block, &freeze_out);

  //printf("adc->calibration_freeze=%d, calibration_freeze=%d, freeze_out.CalFrozen=%d\n", 
  //       adc->calibration_freeze, calibration_freeze, freeze_out.CalFrozen);

  if(calibration_freeze != !!freeze_out.CalFrozen)
    {
      printf("Calibration freeze change failed for %s. Was %s failed to change to %s now at %s.\n", adc_names_sw(adc_number_sw), adc->calibration_freeze ? "frozen" : "operating", calibration_freeze ? "frozen" : "operating", freeze_out.CalFrozen ? "frozen" : "operating");

      global_entry_rejected_beep();
    }

  if(adc->calibration_freeze != !!freeze_out.CalFrozen)
    {
      printf("Calibration freeze changed for %s from %s to %s.\n", adc_names_sw(adc_number_sw), adc->calibration_freeze ? "frozen" : "operating", calibration_freeze ? "frozen" : "operating");
      adc->calibration_freeze = !!freeze_out.CalFrozen;
      //global_entry_accepted_beep();
    }


  XRFdc_SetCalibrationMode(RFdcInstPtr, adcs[adc_number_hw].adc_tile, adcs[adc_number_hw].adc_block, calibration_value);
  XRFdc_GetCalibrationMode(RFdcInstPtr, adcs[adc_number_hw].adc_tile, adcs[adc_number_hw].adc_block, &calibration_mode_u8);

  //printf("Old calibration value %d, commanded value %d, final value %d\n", adc->calibration_mode, calibration_value, calibration_mode_u8);

  if(calibration_value != calibration_mode_u8)
    {
      printf("Calibration mode change failed for %s. Was %d failed to change to %d now at %d.\n", adc_names_sw(adc_number_sw), adc->calibration_mode, calibration_value, calibration_mode_u8);
      global_entry_rejected_beep();
    }

  if(adc->calibration_mode != calibration_mode_u8)
    {
      // Changed, and changed correctly
      printf("Calibration mode changed for %s from %d to %d.\n", adc_names_sw(adc_number_sw), adc->calibration_mode, calibration_mode_u8);
      adc->calibration_mode = calibration_mode_u8;
      //global_entry_accepted_beep();
    }

  // Just in case some settings didn't take, update all settings from what the hardware currently is set to,
  // to make sure HW and GUI are never out of sync.
  update_settings_from_hw();
}


bool select_event(select_one*               sender,
		  selection_changed_event   event_data,
		  menu_adc_control*         receiver,
		  int                       receiver_data,
		  int&                      event_return)
{
  receiver->apply_settings();
  receiver->mark_layout_dirty();
  event_return = 0;
  return true;
}


void menu_adc_control::this_selected()
{
  //printf("Graph Menu %p selected.\n", this);
  mark_layout_dirty();
  parent->mark_layout_dirty();
}



menu_adc_control::menu_adc_control(adc_info& info, color bg)
  : menu(adc_names_hw(info.adc_number_hw), bg)
{
  this->selectable = true;
  this->adc = &info;
  
  //wgc = hwalloc<white_gauss_control> (0xA0050000);
    
  if(info.control)
    {
      printf("Creating a menu_adc_control to control a ADC that already has menu.\n");
      exit(20);
    }

  info.control = this;

  //adc->adc_control = hwalloc<swept_spread_phase_generator> (info.hw_address);

  color c = adc_colors_hw(info.adc_number_hw);

  TEXTCOLOR      = scale(c, REGIONCOLOR, 1.0);
  SELECTEDCOLOR  = scale(c, REGIONCOLOR, 1.0);
  TAGCOLOR       = scale(c, REGIONCOLOR, 0.5);

  nyquist_button              = new select_one("Nyquist",    REGIONCOLOR);
  dither_button               = new select_one("Dither",     REGIONCOLOR);
  calibration_mode_button     = new select_one("Cal Mode",   REGIONCOLOR);
  calibration_freeze_button   = new select_one("Cal Freeze", REGIONCOLOR);

  nyquist_button->add("Zone 1",  (adc->nyquist_zone==1), false);
  nyquist_button->add("Zone 2",  (adc->nyquist_zone==2), false);

  dither_button->add("Disabled",  (adc->dither_enabled==XRFDC_DITH_DISABLE), false);
  dither_button->add("Enabled",   (adc->dither_enabled==XRFDC_DITH_ENABLE),  false);

  if(XRFdc_Version >= XRFDC_GEN3)
    calibration_mode_button->add("Auto", (adc->calibration_mode==0), false);
  calibration_mode_button->add("Mode 1", (adc->calibration_mode==1), false);
  calibration_mode_button->add("Mode 2", (adc->calibration_mode==2), false);

  calibration_freeze_button->add("Operating",  !adc->calibration_freeze,  false);
  calibration_freeze_button->add("Frozen",     adc->calibration_freeze,  false);
  
  select_trigger = new connection<select_one, selection_changed_event, menu_adc_control, int, int>(this, 0, select_event);

  nyquist_button            -> add_receiver(select_trigger);
  dither_button             -> add_receiver(select_trigger);
  calibration_mode_button   -> add_receiver(select_trigger);
  calibration_freeze_button -> add_receiver(select_trigger);
  
  //color text_color      = scale(WHITE, REGIONCOLOR, 1.0);
  //color highlight_color = scale(WHITE, REGIONCOLOR, 0.5);

  multiwindow::add(nyquist_button);
  multiwindow::add(dither_button);
  multiwindow::add(calibration_mode_button);
  multiwindow::add(calibration_freeze_button);

  apply_settings();
}


menu_adc_control::~menu_adc_control()
{
  delete select_trigger;

  //wgc->shift_right[adc->adc_number] = 0;
  //
  //hwfree(wgc);
  //hwfree(adc->adc_control);
  
  multiwindow::remove(nyquist_button);
  multiwindow::remove(dither_button);
  multiwindow::remove(calibration_mode_button);
  multiwindow::remove(calibration_freeze_button);
}


bool menu_adc_control::handle_event(my_event& me)
{
  bool handled = multiwindow::handle_event(me);
  return handled;
}


void menu_adc_control::draw_dirty()
{
  if(!dirty)
    return;
  dirty = false;

  if(nyquist_button->dirty)
    {
      nyquist_button -> draw_dirty();
    }
  if(dither_button->dirty)
    {
      dither_button -> draw_dirty();
    }
  if(calibration_mode_button->dirty)
    {
      calibration_mode_button -> draw_dirty();
    }
  if(calibration_freeze_button->dirty)
    {
      calibration_freeze_button -> draw_dirty();
    }
}


void menu_adc_control::draw_dynamic()
{
  nyquist_button->draw_dynamic();
  dither_button->draw_dynamic();
  calibration_mode_button->draw_dynamic();
  calibration_freeze_button->draw_dynamic();
}


void menu_adc_control::layout()
{
  float designed_height = 1080.0;
  float designed_width  = 1920.0;
  display* d = get_display();
  float ratio_x = d->width / designed_width;
  float ratio_y = d->height / designed_height;
  float ratio = (ratio_x<ratio_y) ? ratio_x : ratio_y;
  extern bool ZCU111;
  
  top_margin            = limit(designed_height * 2/100, 5, 13) * ratio;
  bottom_margin         = top_margin;  
  left_margin           = top_margin;
  right_margin          = top_margin;

  mw_margin             = top_margin;

  tag_top_margin        = top_margin/2;
  tag_bottom_margin     = top_margin/2;
  tag_left_margin       = top_margin;
  tag_right_margin      = top_margin;
  
  selection_margin      = top_margin/5;

  tag_left_offset       = left_margin + tag_left_margin;


  nyquist_button_x_offset = 2 * top_margin;
  nyquist_button_y_offset = 2 * top_margin;

  int num_buttons = 4;
  
  int nyquist_button_desired_height   = (designed_height*ratio - 2*top_margin*num_buttons - bottom_margin)/num_buttons;

  nyquist_button->calculate_width_and_height(nyquist_button_width, nyquist_button_height, nyquist_button_desired_height, ratio);


  dither_button_x_offset = 2 * top_margin;
  dither_button_y_offset = nyquist_button_y_offset + nyquist_button_height + 2 * top_margin;

  int dither_button_desired_height   = nyquist_button_desired_height;

  dither_button->calculate_width_and_height(dither_button_width, dither_button_height, dither_button_desired_height, ratio);


  calibration_mode_button_x_offset = 2 * top_margin;
  calibration_mode_button_y_offset = dither_button_y_offset + dither_button_height + 2 * top_margin;

  int calibration_mode_button_desired_height   = nyquist_button_desired_height;

  calibration_mode_button->calculate_width_and_height(calibration_mode_button_width, calibration_mode_button_height, calibration_mode_button_desired_height, ratio);


  calibration_freeze_button_x_offset = 2 * top_margin;
  calibration_freeze_button_y_offset = calibration_mode_button_y_offset + calibration_mode_button_height + 2 * top_margin;

  int calibration_freeze_button_desired_height   = nyquist_button_desired_height;

  calibration_freeze_button->calculate_width_and_height(calibration_freeze_button_width, calibration_freeze_button_height, calibration_freeze_button_desired_height, ratio);


  
  int button_width  = max(max(max(nyquist_button_width, dither_button_width), calibration_mode_button_width), calibration_freeze_button_width);
  int button_height = calibration_freeze_button_y_offset + calibration_freeze_button_height;

  
  clear(bgcolor);
    
  int circle_width = 20 * ratio;


  /////////////////////////////////////////////////////////////////////////////////////////
  //
  //  Info text
  //
  
  color textcolors[MAX_NUM_ADCS+1] = { WHITE, RED, GREEN, BLUE, ORANGE, BLUEGREEN, PURPLE, YELLOW, GREY };

  //
  // Saved as examples of multi colors
  //
  // "Signal Generator Demo\x0B (Forthcoming; now 10MHz sine)",
  // "\x0C" "Full version is for sale.\x0B (Forthcoming)\x0C  Also ask about:",

  int line_height = 40*ratio;
  
  set_text_size(line_height);

  char line1[200] = "";
  char line2[200] = "";
  char line3[200] = "";
  char line4[200] = "";
  char line5[200] = "";
  char line6[200] = "";
  char line7[200] = "";
  char line8[200] = "";
  char line9[200] = "";
  char line10[200] = "";
  char balun_string[100] = "";

  if(adc->adc_number_hw<2)
    sprintf(balun_string, "a low-frequency balun");
  else if(adc->adc_number_hw<4)
    sprintf(balun_string, "a high-frequency balun");
  else
    sprintf(balun_string, "no balun");

  int adc_number_sw = hw_adc_to_sw_adc(adc->adc_number_hw);

  int usual_dac = 7 - adc_number_sw;

  float low_freq_MHz  = (adc->nyquist_zone==1) ? 0.0                 : sampling_rate_MHz/2;
  float high_freq_MHz = (adc->nyquist_zone==1) ? sampling_rate_MHz/2 : sampling_rate_MHz;

  
  sprintf(line1,  "%c%s%c is configured for Nyquist", 11+adc_number_sw, adc_names_sw(adc_number_sw), 10);
  sprintf(line2,  "Zone %d (\"Band %d\", %g to %g MHz).", adc->nyquist_zone, adc->nyquist_zone, low_freq_MHz, high_freq_MHz);

  if(ZCU111)
    {
      sprintf(line3,  "On the XM500, it has %s.", balun_string);
      sprintf(line4,  "Because of the XM500 configuration,");
      sprintf(line5,  "when looped back it is usually");
      sprintf(line6,  "connected to %cDAC %d%c.", 11+usual_dac, usual_dac, 10);
      sprintf(line7,  "Looping it back requires DC Blocks.");
      sprintf(line8,  "XM500 ADC/ADC connections are illustrated");
      sprintf(line9,  "in the document \"XM500 Connections\"");
      sprintf(line10, "found in \"Documents\" on the main menu.");
    }
    
  int line1_width  = calculate_text_width(line1);
  int line2_width  = calculate_text_width(line2);
  int line3_width  = calculate_text_width(line3);
  int line4_width  = calculate_text_width(line4);
  int line5_width  = calculate_text_width(line5);
  int line6_width  = calculate_text_width(line6);
  int line7_width  = calculate_text_width(line7);
  int line8_width  = calculate_text_width(line8);
  int line9_width  = calculate_text_width(line9);
  int line10_width = calculate_text_width(line10);

  int line_width = max(max(max(max(max(max(max(max(max(line1_width, line2_width), line3_width), line4_width), line5_width), line6_width), line7_width), line8_width), line9_width), line10_width);

  int rect_width = button_width + 2 * top_margin + line_width;


  int explan_text_height = (adc_number_sw>=4) ? 16*line_height : 14*line_height;

  int maxheight   = max(explan_text_height, button_height);
  int rect_height = maxheight+top_margin+bottom_margin;

  // Shouldn't need this.
  if(rect_height>height-circle_width)
    rect_height = height-circle_width;
  
  //
  //
  /////////////////////////////////////////////////////////////////////////////////////////
  

  draw_rect_with_curved_edges(REGIONCOLOR,
			      left_margin,
			      top_margin,
			      rect_width,
			      rect_height,
			      circle_width);


  nyquist_button->resize(nyquist_button_x_offset, nyquist_button_y_offset, nyquist_button_width, nyquist_button_height);
  nyquist_button->layout();

  dither_button->resize(dither_button_x_offset, dither_button_y_offset, dither_button_width, dither_button_height);
  dither_button->layout();

  calibration_mode_button->resize(calibration_mode_button_x_offset, calibration_mode_button_y_offset, calibration_mode_button_width, calibration_mode_button_height);
  calibration_mode_button->layout();

  calibration_freeze_button->resize(calibration_freeze_button_x_offset, calibration_freeze_button_y_offset, calibration_freeze_button_width, calibration_freeze_button_height);
  calibration_freeze_button->layout();


  //int display_width  = get_display()->width;
  //int display_height = get_display()->height;  

  int text_x_offset = 6 * top_margin + button_width;
  int text_y_offset = nyquist_button_y_offset + 2*top_margin;

  draw_multicolored_text(line1,
			 textcolors,
			 text_x_offset,
			 text_y_offset,
			 DRAW_TEXT_X_LEFT|DRAW_TEXT_Y_TOP|DRAW_TEXT_ROTATE_0);
  text_y_offset += line_height * 5/4;
	  
  draw_multicolored_text(line2,
			 textcolors,
			 text_x_offset,
			 text_y_offset,
			 DRAW_TEXT_X_LEFT|DRAW_TEXT_Y_TOP|DRAW_TEXT_ROTATE_0);
  text_y_offset += line_height * 5/4;
	  
  draw_multicolored_text(line3,
			 textcolors,
			 text_x_offset,
			 text_y_offset,
			 DRAW_TEXT_X_LEFT|DRAW_TEXT_Y_TOP|DRAW_TEXT_ROTATE_0);
  text_y_offset += line_height * 2;
	  
  draw_multicolored_text(line4,
			 textcolors,
			 text_x_offset,
			 text_y_offset,
			 DRAW_TEXT_X_LEFT|DRAW_TEXT_Y_TOP|DRAW_TEXT_ROTATE_0);
  text_y_offset += line_height * 5/4;
	  
  draw_multicolored_text(line5,
			 textcolors,
			 text_x_offset,
			 text_y_offset,
			 DRAW_TEXT_X_LEFT|DRAW_TEXT_Y_TOP|DRAW_TEXT_ROTATE_0);
  text_y_offset += line_height * 5/4;
	  
  draw_multicolored_text(line6,
			 textcolors,
			 text_x_offset,
			 text_y_offset,
			 DRAW_TEXT_X_LEFT|DRAW_TEXT_Y_TOP|DRAW_TEXT_ROTATE_0);
  text_y_offset += line_height * 2;


  if(adc_number_sw>=4)
    {
      draw_multicolored_text(line7,
			     textcolors,
			     text_x_offset,
			     text_y_offset,
			     DRAW_TEXT_X_LEFT|DRAW_TEXT_Y_TOP|DRAW_TEXT_ROTATE_0);
      text_y_offset += line_height * 2;
    }
	  
  draw_multicolored_text(line8,
			 textcolors,
			 text_x_offset,
			 text_y_offset,
			 DRAW_TEXT_X_LEFT|DRAW_TEXT_Y_TOP|DRAW_TEXT_ROTATE_0);
  text_y_offset += line_height * 5/4;
	  
  draw_multicolored_text(line9,
			 textcolors,
			 text_x_offset,
			 text_y_offset,
			 DRAW_TEXT_X_LEFT|DRAW_TEXT_Y_TOP|DRAW_TEXT_ROTATE_0);
  text_y_offset += line_height * 5/4;
	  
  draw_multicolored_text(line10,
			 textcolors,
			 text_x_offset,
			 text_y_offset,
			 DRAW_TEXT_X_LEFT|DRAW_TEXT_Y_TOP|DRAW_TEXT_ROTATE_0);
  text_y_offset += line_height * 5/4;

  layout_dirty = false;
  dirty = true;
}
