
// SPDX-License-Identifier: CC-BY-NC-SA-4.0
//
// Copyright (C) 2025 Bit by Bit Signal Processing LLC  (https://bxbsp.com)
//
// This work is placed under the "Creative Commons Attribution
// NonCommercial ShareAlike 4.0 International" license, known
// by the shortened acronym "CC-BY-NC-SA-4.0".
//
// This work is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//
// A CC-BY-NC-SA-4.0 license allows you to use, distribute, and modify
// this work, so long as such uses are non-commercial in nature,
// so long as any derived works are offered on the same terms,
// and so long as attribution is given to the original author.
// For further details, see the Creative Commons License
// "CC-BY-NC-SA-4.0".
//
// You should have received a copy of the CC-BY-NC-SA-4.0 license
// along with this work. If not, see
// <https://creativecommons.org/licenses/by-nc-sa/4.0/>.
//


#include "menu_config.hh"
#include "display.hh"
#include "are_you_sure.hh"
#include <string.h>
#include "data_update.hh"
#include "menu_adc_control.hh"
#include "menu_dac_control.hh"
#include "xrfdc.h"
#include "menu_graph_select.hh"
#include "graph_data_base.hh"

axis_scale adc_freq_list[] =
  {
    //    { TYPE_DEFAULT,   "kHz",     1000.000,                                  false },
    { TYPE_DEFAULT,   "MHz",        1.000,                                  true  },
    { TYPE_DEFAULT,   "GHz",        0.001,                                  false },
  };
int num_adc_freqs = sizeof(adc_freq_list) / sizeof(adc_freq_list[0]);

static int limit(int v, int minval, int maxval)
{
  if(v<minval)
    return minval;
  if(v>maxval)
    return maxval;

  return v;
}


bool number_event(number_box*               sender,
		  double                    value,
		  menu_config*              receiver,
		  int                       receiver_data,
		  int&                      event_return)
{
  receiver->number_event();
  
  event_return = 0;
  return true;
}

bool restart_action(multiselect*              sender,
		    selection_changed_event   event_data,
		    menu_config*              receiver,
		    int                       receiver_data,
		    int&                      event_return)
{
  //receiver->mark_dirty();

  int config_number = receiver->config_select->get_item_number_selected();
  
  printf("menu_config restart button push\n");
  fflush(stdout);

  global_entry_accepted_beep();

  usleep(100000);  // Wait 0.1 second to hear the beep.  Gives confirmation of the push.
  
  extern int     main_argc;
  extern char**  main_argv;

  char* args[100];

  int i;
  for(i=0; i<main_argc; i++)
    args[i] = main_argv[i];

  args[i] = (char*)"-config";
  char num[10];
  sprintf(num, "%d", config_number);
  args[i+1] = num;
  args[i+2] = 0;


  for(int i=0; i<10; i++)
    printf("******************************************************\n");
  
  printf("Terminating program by exec-ing command: \"");
  for(i=0; args[i]; i++)
    printf(" %s", args[i]);
  printf("\"\n");
  fflush(stdout);

  int fdlimit = (int)sysconf(_SC_OPEN_MAX);
  for (int i = STDERR_FILENO + 1; i < fdlimit; i++) close(i);
  
  execvp(main_argv[0], args);
  
  //receiver->mark_layout_dirty();
  //receiver->parent->mark_layout_dirty();
  //event_return = 0;
  
  return true;
}


void in_spec_question_function(void* data, int selection)
{
  menu_config* mc = (menu_config*)data;

  //printf("in inexact_question_function(), mc=%p\n", mc);
  //fflush(stdout);
    
  if(selection!=1)
    {
      // Abort
      printf("Out-of-spec frequency %.6f rejected by user.\n", mc->closest_ADC_frequency_MHz);
      mc->closest_ADC_frequency_MHz = sampling_rate_MHz;
      mc->ADC_frequency_MHz = sampling_rate_MHz;
      mc->mark_layout_dirty();
      mc->parent->mark_layout_dirty();
      global_entry_rejected_beep();
    }
  else
    {
      bool frequency_above_spec = (mc->closest_ADC_frequency_MHz > allowable_configurations->configs[current_config_number].spec_max_adc_sample_hz * 1.0e-6);
      if(frequency_above_spec)
	printf("Out-of-spec frequency %.6f accepted by user.\n", mc->closest_ADC_frequency_MHz);	  
 
      global_entry_accepted_beep();

      char temp[1000];
      
      if(allowable_configurations->configs[current_config_number].adc_multiplier != 1 ||
	 allowable_configurations->configs[current_config_number].dac_multiplier != 1)
	{
	  double base_freq = mc->closest_ADC_frequency_MHz / allowable_configurations->configs[current_config_number].adc_multiplier;
	  
	  sprintf(temp, "%s %f %d %d",
		  (char*)allowable_configurations->configs[current_config_number].clock_config_command,
		  base_freq,
		  allowable_configurations->configs[current_config_number].adc_multiplier,
		  allowable_configurations->configs[current_config_number].dac_multiplier);
	}
      else
	{
	  sprintf(temp, "%s %.6f %.6f",
		  (char*)allowable_configurations->configs[current_config_number].clock_config_command,
		  mc->closest_ADC_frequency_MHz,
		  mc->closest_ADC_frequency_MHz);
	}


      printf("\n\n***********************************************************************************\n\n");
      
      display* d = mc->get_display();

      d->draw_ephemeral_alert("Configuring clocks", 40, WHITE, GREY3);

	
      int ret = system(temp);
      
      printf("Command \"%s\" returned %d\n", temp, ret);



      //#ifdef NOTDEF
      printf("Adjusting graphs for the new sampling rate, old fs=%fMHz, new fs=%fMHz\n", sampling_rate_MHz, mc->closest_ADC_frequency_MHz);

      for(ListElement<window>* le=mc->graph_menu->subwindows->first()->next(); le; le=le->next())
	{
	  menu_graph_select* mgs = (menu_graph_select*)le->data();

	  double xmin = mgs->gc.xmin;
	  double xmax = mgs->gc.xmax;

	  if(mgs->gc.type!=TYPE_TIME)
	    {
	      if(xmin>-2.0 && xmin<2.0 && xmax>sampling_rate_MHz/2-2.0 && xmax<sampling_rate_MHz/2+2.0 )
		{
		  printf("Graph has xmin=%f, xmax=%f; full First Nyquist.  Changing to 0.0, %f.\n", xmin, xmax, mc->closest_ADC_frequency_MHz/2);
		  
		  mgs->gc.xmin = 0.0;
		  mgs->gc.xmax = mc->closest_ADC_frequency_MHz/2;
		}
	      else if(xmin>sampling_rate_MHz/2-2.0 && xmin<sampling_rate_MHz/2+2.0 && xmax>sampling_rate_MHz-2.0 && xmax<sampling_rate_MHz+2.0 )
		{
		  printf("Graph has xmin=%f, xmax=%f; full Second Nyquist.  Changing to %f, %f.\n", xmin, xmax, mc->closest_ADC_frequency_MHz/2, mc->closest_ADC_frequency_MHz);
		  
		  mgs->gc.xmin = mc->closest_ADC_frequency_MHz/2;
		  mgs->gc.xmax = mc->closest_ADC_frequency_MHz;
		}
	      else
		{
		  printf("Graph has xmin=%f, xmax=%f.  Not changing.\n", xmin, xmax);
		}
	    }

	}
      //#endif


      
      usleep(10000);  // Wait 0.01 second for clock to settle before reconfiguring ADCs and DACs

      allowable_configurations->configs[current_config_number].set_adc_sampling_rate_hz(mc->closest_ADC_frequency_MHz*1e6);
      allowable_configurations->configs[current_config_number].set_dac_sampling_rate_hz(mc->closest_ADC_frequency_MHz*1e6);
      x_axis_scale_list_init();
        
      mc->ADC_frequency_MHz = mc->closest_ADC_frequency_MHz;
      
      //
      // Reset the ADCs and DACs; causes them to rerun calibration
      //
      extern XRFdc *RFdcInstPtr;
      
      // u32 Status = XRFdc_Reset(RFdcInstPtr, XRFDC_ADC_TILE, -1);  // reset all ADC tiles.

      u32 Status = XRFdc_StartUp(RFdcInstPtr, XRFDC_ADC_TILE, -1);  // reset all ADC tiles, without clearing all settings
      
      if(Status != XRFDC_SUCCESS)
	{
	  printf("WARNING:  ADC tiles didn't fully reset.\n");
	}
      
      //Status = XRFdc_Reset(RFdcInstPtr, XRFDC_DAC_TILE, -1);  // reset all DAC tiles.

      Status = XRFdc_StartUp(RFdcInstPtr, XRFDC_DAC_TILE, -1);  // reset all DAC tiles, without clearing all settings
      
      if(Status != XRFDC_SUCCESS)
	{
	  printf("WARNING:  DAC tiles didn't fully reset.\n");
	}
      
      // Reapply DAC controls.  These change with change in sampling rate.
      // Also, the tile reset may have messed with the settings.
      
      extern menu_dac_control* dac_controls[MAX_NUM_DACS];
      for(int i=0; i<NUM_DACS; i++)
	{      
	  dac_controls[i]->apply_settings();
	}
      
      // Reapply ADC controls.  The tile reset may have messed with the settings.
      
      extern menu_adc_control* adc_controls[MAX_NUM_ADCS];
      for(int i=0; i<NUM_ADCS; i++)
	{      
	  adc_controls[i]->apply_settings();
	}

      XRFdc_IPStatus IPStatus;
      
      u32 IPStatusReturn = XRFdc_GetIPStatus(RFdcInstPtr, &IPStatus);
      printf("Results from XRFdc_GetIPStatus (Which has return value IPStatusReturn=%d):\n", IPStatusReturn);
      printf("   State is 0x%08X\n", (int)IPStatus.State);
      for(int i=0; i<2; i++)
	{
	  printf("   DAC Tile %d status:\n", i);
	  printf("      is_Enabled       = %d\n",     (int)IPStatus.DACTileStatus[i].IsEnabled);
	  printf("      TileState        = %d\n",     (int)IPStatus.DACTileStatus[i].TileState);
	  printf("      BlockStatusMask  = 0x%02X\n", (int)IPStatus.DACTileStatus[i].BlockStatusMask);
	  printf("      PowerUpState     = %d\n",     (int)IPStatus.DACTileStatus[i].PowerUpState);
	  printf("      PLLState         = %d\n",     (int)IPStatus.DACTileStatus[i].PLLState);
	}
      for(int i=0; i<4; i++)
	{
	  printf("   ADC Tile %d status:\n", i);
	  printf("      is_Enabled       = %d\n",     (int)IPStatus.ADCTileStatus[i].IsEnabled);
	  printf("      TileState        = %d\n",     (int)IPStatus.ADCTileStatus[i].TileState);
	  printf("      BlockStatusMask  = 0x%02X\n", (int)IPStatus.ADCTileStatus[i].BlockStatusMask);
	  printf("      PowerUpState     = %d\n",     (int)IPStatus.ADCTileStatus[i].PowerUpState);
	  printf("      PLLState         = %d\n",     (int)IPStatus.ADCTileStatus[i].PLLState);
	}
    }
}


void inexact_question_function(void* data, int selection)
{
  menu_config* mc = (menu_config*)data;

  //printf("in inexact_question_function(), mc=%p\n", mc);
  //fflush(stdout);
  
  
  if(selection!=1)
    {
      // Abort
      printf("Inexact frequency %.6f rejected by user.\n", mc->closest_ADC_frequency_MHz);
      mc->closest_ADC_frequency_MHz = sampling_rate_MHz;
      mc->ADC_frequency_MHz = sampling_rate_MHz;
      mc->mark_layout_dirty();
      mc->parent->mark_layout_dirty();
      global_entry_rejected_beep();
    }
  else
    {
      // Isn't inexact, or OK being inexact.

      bool inexact = ( fabs(mc->closest_ADC_frequency_MHz - mc->ADC_frequency_MHz) > 1e-6 );
      if(inexact)
	printf("Inexact frequency %.6f accepted by user.\n", mc->closest_ADC_frequency_MHz);      

      bool frequency_above_spec = (mc->closest_ADC_frequency_MHz > allowable_configurations->configs[current_config_number].spec_max_adc_sample_hz * 1.0e-6);
      if(!frequency_above_spec)
	{
	  in_spec_question_function((void*)mc, 1);
	}
      else
	{
	  char temp[200];
	  char temp2[200];
	  sprintf(temp, "DANGER: %.6fMHz exceeds max %.6fMHz.", mc->closest_ADC_frequency_MHz, allowable_configurations->configs[current_config_number].spec_max_adc_sample_hz * 1.0e-6);
	  sprintf(temp2, "Don't change; stay at %.6fMHz.", sampling_rate_MHz);
	  
	  ask_are_you_sure(mc->get_display(),
			   temp,
			   0,
			   "Accept responsibility for any damage, and switch.",
			   0,
			   temp2,
			   in_spec_question_function,
			   (void*)mc);
	  
	}
    }
  
  fflush(stdout);
}


void menu_config::number_event()
{
  char temp[1000];

  if(allowable_configurations->configs[current_config_number].adc_multiplier != 1 ||
     allowable_configurations->configs[current_config_number].dac_multiplier != 1)
    {
      double base_freq = ADC_frequency_MHz / allowable_configurations->configs[current_config_number].adc_multiplier;
      
      sprintf(temp, "%s -check %f %d %d",
	      (char*)allowable_configurations->configs[current_config_number].clock_config_command,
	      base_freq,
	      allowable_configurations->configs[current_config_number].adc_multiplier,
	      allowable_configurations->configs[current_config_number].dac_multiplier);
    }
  else
    {
      sprintf(temp, "%s -check %.6f %.6f",
	      (char*)allowable_configurations->configs[current_config_number].clock_config_command,
	      ADC_frequency_MHz,
	      ADC_frequency_MHz);
    }
  
  FILE* fp = popen(temp, "r");

  double closest_DAC_frequency_MHz;
  
  int num = fscanf(fp, "%lf %lf", &closest_ADC_frequency_MHz, &closest_DAC_frequency_MHz);

  if(num!=2)
    printf("Warning:  Didn't get back good values for selectable frequency.");

  pclose(fp);

  printf("menu_config acivate button push\n");
  fflush(stdout);

  bool inexact = ( fabs(closest_ADC_frequency_MHz - ADC_frequency_MHz) > 1e-6 );
  if(!inexact)
    {
      inexact_question_function((void*)this, 1);
    }
  else
    {
      char temp[200];
      char temp2[200];
      sprintf(temp, "Use %.6fMHz (not requested %.6fMHz)", closest_ADC_frequency_MHz, ADC_frequency_MHz);
      sprintf(temp2, "Don't change; stay at %.6fMHz.", sampling_rate_MHz);

      //printf("menu_config is %p\n", receiver);
      
      ask_are_you_sure(get_display(),
		       "Warning: New Frequency will be inexact.",
		       0,
		       temp,
		       0,
		       temp2,
		       inexact_question_function,
		       (void*)this);
    }

  mark_layout_dirty();
  parent->mark_layout_dirty();
}



bool config_select_receiver(select_one*               sender,
			    selection_changed_event   event_data,
			    menu_config*              receiver,
			    int                       receiver_data,
			    int&                      event_return)
{
  //int config_number = event_data.selection_number;

  //allowable_configurations->set_current_config(config_number);

  //new are_you_sure(receiver->get_display(),
  //		   "Are You Sure?",
  //		   0,
  //		   "Reconfigure the clocks, reload the FPGA, restart the program",
  //		   0,
  //		   "Don't do this",
  //		   confirm_selection,
  //		   &config_number);
  //global_entry_rejected_beep();

  // Changes the display.  Nothing actually happens until an action button is pushed.
  receiver->mark_layout_dirty();
  receiver->parent->mark_layout_dirty();

  event_return = 1;
  return 0;
}


void menu_config::this_selected()
{
  //printf("Graph Menu %p selected.\n", this);
  mark_layout_dirty();
  parent->mark_layout_dirty();
}



menu_config::menu_config(const char* tagname, menu* graph_menu, color bg)
  : menu(tagname, bg)
{
  this->selectable = true;
  this->graph_menu = graph_menu;

  config_select = new select_one("Sampling Configuration", REGIONCOLOR);
  config_select->TITLECOLOR = BLUE;
  
  for(int i=0; i<allowable_configurations->num_configs; i++)
    {
      bool selected = (current_config_number==i);
      config_select->add(allowable_configurations->configs[i].name, selected, false);
    }


  ADC_frequency_MHz = sampling_rate_MHz;
  closest_ADC_frequency_MHz = sampling_rate_MHz;

  c_config_select = new connection<select_one, selection_changed_event, menu_config, int, int>(this, 0, config_select_receiver);


  config_select -> add_receiver(c_config_select);

  restart_button = new button("Restart",       // button_name
			      20,              // font_size
			      10,              // x_margin
			      10,              // y_margin
			      RED,             // text color
			      REDRED6,         // fg color
			      REGIONCOLOR);    // bg color

  c_restart = new connection<multiselect, selection_changed_event, menu_config, int, int>(this, 0, restart_action);
  c_number  = new connection<number_box, double, menu_config, int, int>(this, 0, ::number_event);
  
  color TEXTCOLOR      = scale(WHITE, REGIONCOLOR, 1.0);
  color SELECTEDCOLOR  = scale(WHITE, REGIONCOLOR, 1.0);

  double smallest_freq = 500.0;
  double largest_freq  = 7500.0;

  ADC_freq = new number_box("Sampling Rate",
			    &ADC_frequency_MHz,
			    smallest_freq,
			    largest_freq,
			    adc_freq_list,
			    num_adc_freqs,
			    TYPE_DEFAULT,  // type
			    35, // font_size
			    20, // x_margin
			    20, // y_margin
			    TEXTCOLOR,
			    SELECTEDCOLOR,
			    REGIONCOLOR);

  ADC_freq -> add_receiver(c_number);

  restart_button -> add_receiver(c_restart);

  multiwindow::add(config_select);
  multiwindow::add(restart_button);
  multiwindow::add(ADC_freq);
}


menu_config::~menu_config()
{
  delete c_config_select;

  multiwindow::remove(config_select);
}


bool menu_config::handle_event(my_event& me)
{
  bool handled = multiwindow::handle_event(me);
  return handled;
}


void menu_config::draw_dirty()
{
  if(!dirty)
    return;
  dirty = false;
  
  if(config_select->dirty)
    config_select -> draw_dirty();

  if(restart_button->dirty)
    restart_button -> draw_dirty();

  if(ADC_freq->dirty)
    ADC_freq -> draw_dirty();
}


void menu_config::draw_dynamic()
{
  //config_select -> draw_dynamic();
  multiwindow::draw_dynamic();
}


void menu_config::layout()
{
  float designed_height = 1080.0;
  float designed_width  = 1920.0;
  display* d = get_display();
  float ratio_x = d->width / designed_width;
  float ratio_y = d->height / designed_height;
  float ratio = (ratio_x<ratio_y) ? ratio_x : ratio_y;
  
  top_margin            = limit(designed_height * 2/100, 5, 13) * ratio;
  bottom_margin         = top_margin;  
  left_margin           = top_margin;
  right_margin          = 2*top_margin;

  mw_margin             = top_margin;

  tag_top_margin        = top_margin/2;
  tag_bottom_margin     = top_margin/2;
  tag_left_margin       = top_margin;
  tag_right_margin      = top_margin;
  
  selection_margin      = top_margin/5;

  tag_left_offset       = left_margin + tag_left_margin;

  int line_height = 40*ratio;

  set_text_size(line_height);

  config_select_sel_x_offset = 2 * top_margin;
  config_select_sel_y_offset = 2 * top_margin + selected_y_offset;

  int config_select_sel_desired_height = designed_height - top_margin/ratio - bottom_margin/ratio;

  config_select->calculate_width_and_height(config_select_sel_width, config_select_sel_height, config_select_sel_desired_height, ratio);

  
    
  int circle_width = 20 * ratio;


  char line[200] = "";
  const char* description = (char*)allowable_configurations->configs[config_select->get_item_number_selected()].description;

  set_text_size(line_height);

  constexpr int max_line_chars = 60;
  int lines = 0;
  int char_width =  calculate_text_width("WWW");

  int text_y_offset = 2*top_margin + selected_y_offset;
  int j=0;
  for(int i=0;; i++)
    {
      char c = description[i];
      
      if(j==max_line_chars)
	{
	  line[j++] = c;
	  line[j] = 0;
	  c = '\n';
	}
      
      if(c=='\n' || c==0)
	{
	  line[j] = 0;
	  
	  lines++;
	  int this_char_width =  calculate_text_width(line);
	  if(this_char_width>char_width)
	    char_width = this_char_width;
	  j = 0;
	}
      else
	{
	  line[j++] = c;
	}
      
      if(c==0)
	break;
    }

  //printf("lines=%d, description=\"%s\".\n", lines, description);

  const char* line_1 = "Restart to change to this configuration.";
  const char* line_2 = "This resets the clocks and reloads the FPGA PL fabric.";
  const char* line_3 = "This exits and restarts this program (about 5 seconds).";
  const char* line_4 = "If connected via the web, you will need to reconnect.";

  {
    int this_char_width =  calculate_text_width(line_1);
    if(this_char_width>char_width)
      char_width = this_char_width;
  }
  {
    int this_char_width =  calculate_text_width(line_2);
    if(this_char_width>char_width)
      char_width = this_char_width;
  }
  {
    int this_char_width =  calculate_text_width(line_3);
    if(this_char_width>char_width)
      char_width = this_char_width;
  }
  {
    int this_char_width =  calculate_text_width(line_4);
    if(this_char_width>char_width)
      char_width = this_char_width;
  }
  
  int maxheight  = max(config_select_sel_height, lines*line_height*5/4);;
  int maxheight2 = max(config_select_sel_height, lines*line_height*5/4);;

  //if(top_margin==0) printf("x is zero");
  //if(config_select_sel_width==0) printf("y is zero");
  //if(bottom_margin==0) printf("h is zero");
  //if(maxheight==0) printf("h is zero");

  int selected_config_number = config_select->get_item_number_selected();

  if(selected_config_number != current_config_number && allowable_configurations->configs[selected_config_number].restart_app)
    {
      maxheight += 5*line_height*5/4;
    }

  int ADC_freq_desired_height = 7*line_height;
  int ADC_freq_width;
  int ADC_freq_height;
  ADC_freq->calculate_width_and_height(ADC_freq_width, ADC_freq_height, ADC_freq_desired_height, ratio);
  if(allowable_configurations->configs[selected_config_number].selectable_sampling_rate)
    {
      maxheight += ADC_freq_height + 3 * line_height;
    }

  //  bool frequency_above_spec = (ADC_frequency_MHz > allowable_configurations->configs[selected_config_number].spec_max_adc_sample_hz * 1.0e-6);
  //  bool inexact = ( fabs(closest_ADC_frequency_MHz - ADC_frequency_MHz) > 1e-6 );
  
  int restart_button_width;
  int restart_button_height;
  int restart_button_desired_height = line_height;

  restart_button->font_height = line_height;

  restart_button->calculate_width_and_height(restart_button_width, restart_button_height, restart_button_desired_height, ratio);

  clear(bgcolor);

  draw_rect_with_curved_edges(REGIONCOLOR,
			      left_margin,
			      top_margin + selected_y_offset - top_margin,
			      config_select_sel_width + 5 * top_margin + char_width + right_margin,
			      maxheight+2*top_margin+bottom_margin,
			      circle_width);

  if(selected_config_number != current_config_number)
    {
      restart_button->resize(config_select_sel_x_offset + left_margin + config_select_sel_width/2 - restart_button_width/2,
			     (config_select_sel_y_offset + config_select_sel_height + selected_y_offset + maxheight + top_margin)/2 - restart_button_height/2,
			     restart_button_width,
			     restart_button_height);
    }
  else
    {
      restart_button->resize(d->width+1, d->height+1, restart_button_width, restart_button_height);
    }
  
  restart_button->layout();

  config_select->resize(config_select_sel_x_offset, config_select_sel_y_offset, config_select_sel_width, config_select_sel_height);
  config_select->layout();


  set_text_size(line_height);


  if(allowable_configurations->configs[selected_config_number].selectable_sampling_rate && selected_config_number == current_config_number)
    {
      const char* seltxt = "Select a desired sampling rate";
      int seltxtwidth = calculate_text_width(seltxt);
      draw_text(seltxt,
		BLUE,
		6 * left_margin + config_select_sel_x_offset + config_select_sel_width,
		maxheight2 + selected_y_offset + line_height * 3/4,
		DRAW_TEXT_X_LEFT|DRAW_TEXT_Y_TOP|DRAW_TEXT_ROTATE_0);
      int numboxoffset = (seltxtwidth-ADC_freq_width)/2;
      ADC_freq->resize(numboxoffset + 6 * left_margin + config_select_sel_x_offset + config_select_sel_width,
		       maxheight2 + selected_y_offset + line_height * 9/4,
		       ADC_freq_width, ADC_freq_height);
    }
  else
    {
      ADC_freq->resize(d->width+1, d->height+1, ADC_freq_width, ADC_freq_height);
    }

  ADC_freq->layout();
  

  j=0;
  for(int i=0;; i++)
    {
      char c = description[i];
      
      if(j==max_line_chars)
	{
	  line[j++] = c;
	  line[j] = 0;
	  c = '\n';
	}
      
      if(c=='\n' || c==0)
	{
	  line[j] = 0;
	  
	  draw_text(line,
		    TEXTCOLOR,
		    3 * left_margin + config_select_sel_x_offset + config_select_sel_width,
		    text_y_offset,
		    DRAW_TEXT_X_LEFT|DRAW_TEXT_Y_TOP|DRAW_TEXT_ROTATE_0);
	  text_y_offset += line_height * 5/4;
	  j = 0;
	}
      else
	{
	  line[j++] = c;
	}
      
      if(c==0)
	break;
    }


  //int cheight = maxheight2 + selected_y_offset + line_height * 5/4 + line_height * 9 / 4;
  //
  //  if(frequency_above_spec)
  //    {
  //      draw_text("WARNING: Exceeds max spec.",
  //		RED,	
  //		4 * left_margin + config_select_sel_x_offset + config_select_sel_width + ADC_freq_width,
  //		cheight,
  //		DRAW_TEXT_X_LEFT|DRAW_TEXT_Y_TOP|DRAW_TEXT_ROTATE_0);
  //      cheight += line_height;
  //      draw_text("This may cause damage.",
  //		RED,	
  //		4 * left_margin + config_select_sel_x_offset + config_select_sel_width + ADC_freq_width,
  //		cheight,
  //		DRAW_TEXT_X_LEFT|DRAW_TEXT_Y_TOP|DRAW_TEXT_ROTATE_0);
  //      cheight += line_height*3/2;
  //    }
  //
  //  if(inexact)
  //    {
  //      draw_text("WARNING: Frequency isn't exact.",
  //		RED,	
  //		4 * left_margin + config_select_sel_x_offset + config_select_sel_width + ADC_freq_width,
  //		cheight,
  //		DRAW_TEXT_X_LEFT|DRAW_TEXT_Y_TOP|DRAW_TEXT_ROTATE_0);
  //      cheight += line_height;
  //      char text[200];
  //      sprintf(text, "It will be %.6fMHz", closest_ADC_frequency_MHz);
  //      draw_text(text,
  //		RED,	
  //		4 * left_margin + config_select_sel_x_offset + config_select_sel_width + ADC_freq_width,
  //		cheight,
  //		DRAW_TEXT_X_LEFT|DRAW_TEXT_Y_TOP|DRAW_TEXT_ROTATE_0);
  //    cheight += 2*line_height;
  //    }

  
  text_y_offset = maxheight2 + selected_y_offset + line_height * 9/8;
  if(selected_config_number != current_config_number)
    {
      draw_text(line_1,
		RED,
		3 * left_margin + config_select_sel_x_offset + config_select_sel_width,
		text_y_offset,
		DRAW_TEXT_X_LEFT|DRAW_TEXT_Y_TOP|DRAW_TEXT_ROTATE_0);
      text_y_offset += line_height * 5/4;
      draw_text(line_2,
		RED,
		3 * left_margin + config_select_sel_x_offset + config_select_sel_width,
		text_y_offset,
		DRAW_TEXT_X_LEFT|DRAW_TEXT_Y_TOP|DRAW_TEXT_ROTATE_0);
      text_y_offset += line_height * 5/4;
      draw_text(line_3,
		RED,
		3 * left_margin + config_select_sel_x_offset + config_select_sel_width,
		text_y_offset,
		DRAW_TEXT_X_LEFT|DRAW_TEXT_Y_TOP|DRAW_TEXT_ROTATE_0);
      text_y_offset += line_height * 5/4;
      draw_text(line_4,
		RED,
		3 * left_margin + config_select_sel_x_offset + config_select_sel_width,
		text_y_offset,
		DRAW_TEXT_X_LEFT|DRAW_TEXT_Y_TOP|DRAW_TEXT_ROTATE_0);
      text_y_offset += line_height * 5/4;
    }
  
  layout_dirty = false;
  dirty = true;
}

