
// SPDX-License-Identifier: CC-BY-NC-SA-4.0
//
// Copyright (C) 2025 Bit by Bit Signal Processing LLC  (https://bxbsp.com)
//
// This work is placed under the "Creative Commons Attribution
// NonCommercial ShareAlike 4.0 International" license, known
// by the shortened acronym "CC-BY-NC-SA-4.0".
//
// This work is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//
// A CC-BY-NC-SA-4.0 license allows you to use, distribute, and modify
// this work, so long as such uses are non-commercial in nature,
// so long as any derived works are offered on the same terms,
// and so long as attribution is given to the original author.
// For further details, see the Creative Commons License
// "CC-BY-NC-SA-4.0".
//
// You should have received a copy of the CC-BY-NC-SA-4.0 license
// along with this work. If not, see
// <https://creativecommons.org/licenses/by-nc-sa/4.0/>.
//


#include <string.h>

#include "menu_dac_control.hh"
#include "hwalloc.hh"
#include "xrfdc.h"
#include "bxbapp_config.hh"
#include "data_update.hh"
#include "display.hh"
#include "RFSoC4x2_BxBDemo1.hh"

extern XRFdc *RFdcInstPtr;

int NUM_DACS = 8;

axis_scale dac_freq_list[] =
  {
    { TYPE_DEFAULT,   "kHz",     1000.000,                                  false },
    { TYPE_DEFAULT,   "MHz",        1.000,                                  true  },
    { TYPE_DEFAULT,   "GHz",        0.001,                                  false },
  };
int num_dac_freqs = sizeof(dac_freq_list) / sizeof(dac_freq_list[0]);


axis_scale dac_ramp_list[] =
  {
    { TYPE_DEFAULT,   "kHz/s",           1000.0,                            false },
    { TYPE_DEFAULT,   "MHz/min",         60,                                false },
    { TYPE_DEFAULT,   "MHz/s",           1.0,                               true  },
  };
int num_dac_ramps = sizeof(dac_ramp_list) / sizeof(dac_ramp_list[0]);


axis_scale dac_mod_list[] =
  {
    { TYPE_DEFAULT,   "kbps",           1000.0,                            false },
    { TYPE_DEFAULT,   "Mbps",           1.0,                               true  },
  };
int num_dac_mods = sizeof(dac_mod_list) / sizeof(dac_mod_list[0]);


uint32_t wgc_enable_value      = 0x80000000;
uint32_t wgc_max_enable_value  = 0xC0000000;
struct white_gauss_control
{
  uint32_t shift_right[8];
};



void menu_dac_control::apply_settings()
{
  int zone;
  u32 zone_u32;
  int nyquist_num = nyquist_button->get_item_number_selected();
  int nyquist_value = nyquist_num + 1;;
  int dac_number = dac->dac_number;

  XRFdc_SetNyquistZone(RFdcInstPtr, XRFDC_DAC_TILE, dacs[dac_number].dac_tile, dacs[dac_number].dac_block, nyquist_value);
  XRFdc_GetNyquistZone(RFdcInstPtr,  XRFDC_DAC_TILE, dacs[dac_number].dac_tile, dacs[dac_number].dac_block, &zone_u32);

  //
  // I've come to believe the reset doesn't help, and may change the Nyquist zone.
  //
  // To be safe, reset DAC after a Nyquist zone change
  //
  //int Status = XRFdc_Reset(RFdcInstPtr, XRFDC_DAC_TILE, dacs[dac_number].dac_tile);
  //
  //if(Status != XRFDC_SUCCESS)
  //  {
  //    printf("WARNING:  DAC tile didn't fully reset after Nyquist zone change.\n");
  //  }

  
  zone = zone_u32;

  bool zone_change = false;
  
  if(dac->nyquist_zone == zone)
    {
      // No change.  This shouldn't really happen, because it shouldn't cause an event that gets here.
    }
  else if(zone==nyquist_value)
    {
      // Changed, and changed correctly
      printf("Nyquist zone changed for %s from %d to %d.\n", dac_names_hw(dac_number), dac->nyquist_zone, zone);
      dac->nyquist_zone = zone;
      zone_change = true;
      //global_entry_accepted_beep();
    }
  else
    {
      // Changed, but it didn't work.
      printf("Nyquist zone change failed for %s. Was %d failed to change to %d.\n", dac_names_hw(dac_number), dac->nyquist_zone, nyquist_value);
      global_entry_rejected_beep();
    }


  if(zone_change)
    {
      float smallest_freq = (dac->nyquist_zone==1) ? 0 : dac_sampling_rate_MHz/2;
      float largest_freq  = (dac->nyquist_zone==1) ? dac_sampling_rate_MHz/2 : dac_sampling_rate_MHz;

      freq_sel -> min_value = smallest_freq;
      freq_sel -> max_value = largest_freq;

      min_freq_sel -> min_value = smallest_freq;
      min_freq_sel -> max_value = largest_freq;

      max_freq_sel -> min_value = smallest_freq;
      max_freq_sel -> max_value = largest_freq;


      min_frequency_MHz = dac_sampling_rate_MHz - min_frequency_MHz;  // Will be reversed correctly below.
      max_frequency_MHz = dac_sampling_rate_MHz - max_frequency_MHz;
      frequency_MHz     = dac_sampling_rate_MHz - frequency_MHz;
    }
  
  
  bool sine_enabled  = enable_button->selected[0];
  bool gauss_enabled = enable_button->selected[1];
  int sweep_num      = sweep_button->get_item_number_selected();
  int modulate_num   = modulate_button->get_item_number_selected();
  int amplitude_num  = amplitude_button->get_item_number_selected();
  int gauss_amplitude_num  = gauss_amplitude_button->get_item_number_selected();

  amplitude_shift = amplitude_num;

  if(sine_enabled && sweep_num==1)
    {
      // Reverse min and max freq if they're reversed

      if(min_frequency_MHz > max_frequency_MHz)
	{
	  double temp = min_frequency_MHz;
	  min_frequency_MHz = max_frequency_MHz;
	  max_frequency_MHz = temp;
	}
    }
  
  if(gauss_enabled)
    wgc->shift_right[dac_number] = (gauss_amplitude_num==0) ? wgc_max_enable_value : wgc_enable_value + (gauss_amplitude_num-1);
  else
    wgc->shift_right[dac_number] = 0;
    
  double frequency_in_zone_MHz     = (dac->nyquist_zone==1) ?     frequency_MHz : dac_sampling_rate_MHz -     frequency_MHz;
  double min_frequency_in_zone_MHz = (dac->nyquist_zone==1) ? min_frequency_MHz : dac_sampling_rate_MHz - min_frequency_MHz;
  double max_frequency_in_zone_MHz = (dac->nyquist_zone==1) ? max_frequency_MHz : dac_sampling_rate_MHz - max_frequency_MHz;

  uint32_t freq_int             = floor(    frequency_in_zone_MHz * (1ll<<32) / dac_sampling_rate_kHz * 1000.0);
  uint32_t freq_frac            = (frequency_in_zone_MHz * 1000.0 - freq_int * dac_sampling_rate_kHz / (1ll<<32)) * (1ll<<32) * (1ll<<32) / dac_sampling_rate_kHz;
  uint32_t min_freq_int         = floor(min_frequency_in_zone_MHz * (1ll<<32) / dac_sampling_rate_MHz + 0.5);
  uint32_t max_freq_int         = floor(max_frequency_in_zone_MHz * (1ll<<32) / dac_sampling_rate_MHz + 0.5);
  uint32_t ramp_rate_int        = floor( ramp_rate_MHz_per_second * (1ll<<32) / dac_sampling_rate_MHz / dac_sampling_rate_MHz / 121.5 + 0.5);
  uint32_t modulation_count_int = floor(dac_sampling_rate_MHz / modulate_rate_Mbps / dac_samples_per_clock + 0.5) - 1;

  if(ramp_rate_int==0)
    ramp_rate_int=1;


  //printf("DAC:  dac_sampling_rate_MHz=%f, frequency_MHz=%f, frequency_in_zone_MHz=%f\n", dac_sampling_rate_MHz, frequency_MHz, frequency_in_zone_MHz);
  
  //printf("**************************************  dac_samples_per_clock=%d\n", dac_samples_per_clock);
  
  //printf("freq=%.12f, freq_int=0x%08x, freq_frac=0x%08x, ", frequency_in_zone_MHz, freq_int, freq_frac);
  
  //
  // Translate back from integer values so the user can see the actual setting achieved.
  //
  frequency_in_zone_MHz         = freq_int * dac_sampling_rate_MHz / (1ll<<32) + freq_frac * dac_sampling_rate_MHz / (1ll<<32) / (1ll<<32);
  min_frequency_in_zone_MHz     = min_freq_int * dac_sampling_rate_MHz / (1ll<<32);
  max_frequency_in_zone_MHz     = max_freq_int * dac_sampling_rate_MHz / (1ll<<32);
  ramp_rate_MHz_per_second      = ramp_rate_int * dac_sampling_rate_MHz * dac_sampling_rate_MHz * 121.5 / (1ll<<32);
  modulate_rate_Mbps            = dac_sampling_rate_MHz / dac_samples_per_clock / ( modulation_count_int + 1);

  //printf("freq_actual=%.12f\n", frequency_in_zone_MHz);

  frequency_in_zone_MHz     = (dac->nyquist_zone==1) ?     frequency_in_zone_MHz : dac_sampling_rate_MHz -     frequency_in_zone_MHz;
  min_frequency_in_zone_MHz = (dac->nyquist_zone==1) ? min_frequency_in_zone_MHz : dac_sampling_rate_MHz - min_frequency_in_zone_MHz;
  max_frequency_in_zone_MHz = (dac->nyquist_zone==1) ? max_frequency_in_zone_MHz : dac_sampling_rate_MHz - max_frequency_in_zone_MHz;

  if(!sine_enabled)
    {
      // Disabled
      dac->dac_control->sine_shift_right   = -1;
      dac->dac_control->freq_min           = 0;
      dac->dac_control->freq_max           = 0;
      dac->dac_control->freq_ramp          = 0;
      dac->dac_control->spread_period      = 0;
      dac->dac_control->spread_code        = 0;
      dac->dac_control->freq_fractional    = 0;
    }
  else
    {
      // Enabled

      if(sweep_num==0)
	{
	  // Constant
	  dac->dac_control->freq_ramp        = 0;
	  dac->dac_control->freq_min         = freq_int;
	  dac->dac_control->freq_max         = freq_int;
	  dac->dac_control->freq_fractional  = freq_frac;
	}
      else
	{
	  // Sweep	  
    	  dac->dac_control->freq_ramp        = ramp_rate_int;
	  dac->dac_control->freq_min         = (dac->nyquist_zone==1) ? min_freq_int : max_freq_int;
	  dac->dac_control->freq_max         = (dac->nyquist_zone==1) ? max_freq_int : min_freq_int;
	  dac->dac_control->freq_fractional  = 0;
	}

      if(modulate_num==0)
	{
	  // No modulation
	  dac->dac_control->spread_period   = 0;
	  dac->dac_control->spread_code     = 0;
	}
      else if(modulate_num==1)
	{
	  // BPSK modulation

	  // Only one PN output every 8 clocks for (SAMPLES_PER_CLOCK==8).  Now also support SAMPLES_PER_CLOCK=10
	  dac->dac_control->spread_period   = modulation_count_int;
	  dac->dac_control->spread_code     = 0x80002001;
	}
      else
	{
	  // No longer valid case, but it can't occur so it doesn't hurt.

	  // Only one PN output every 8 clocks for (SAMPLES_PER_CLOCK==8).  Now also support SAMPLES_PER_CLOCK=10
	  dac->dac_control->spread_period   = modulation_count_int;
	  dac->dac_control->spread_code     = 0x80002001;
	}
      
      dac->dac_control->sine_shift_right = amplitude_shift;
    }
}


static int limit(int v, int minval, int maxval)
{
  if(v<minval)
    return minval;
  if(v>maxval)
    return maxval;

  return v;
}


bool select_event(select_one*               sender,
		  selection_changed_event   event_data,
		  menu_dac_control*         receiver,
		  int                       receiver_data,
		  int&                      event_return)
{
  bool sine_enabled  = receiver->enable_button->selected[0];
  bool gauss_enabled = receiver->enable_button->selected[1];

  if(sine_enabled && gauss_enabled)
    {
      int amplitude_num  = receiver->amplitude_button->get_item_number_selected();
      int gauss_amplitude_num  = receiver->gauss_amplitude_button->get_item_number_selected();

      if(amplitude_num==0)
	receiver->amplitude_button->select(1);

      if(gauss_amplitude_num==0 || gauss_amplitude_num==1)
	receiver->gauss_amplitude_button->select(2);

      receiver->amplitude_button->disabled[0] = true;
      receiver->gauss_amplitude_button->disabled[0] = true;
      receiver->gauss_amplitude_button->disabled[1] = true;
    }
  else
    {
      receiver->amplitude_button->disabled[0] = false;
      receiver->gauss_amplitude_button->disabled[0] = false;
      receiver->gauss_amplitude_button->disabled[1] = false;
    }

  
  receiver->apply_settings();
  receiver->mark_layout_dirty();
  
  event_return = 0;
  return true;
}


bool number_event(number_box*               sender,
		  double                    value,
		  menu_dac_control*         receiver,
		  int                       receiver_data,
		  int&                      event_return)
{
  receiver->apply_settings();
  
  event_return = 0;
  return true;
}


void menu_dac_control::this_selected()
{
  //printf("Graph Menu %p selected.\n", this);
  mark_layout_dirty();
  parent->mark_layout_dirty();
}



menu_dac_control::menu_dac_control(dac_info& info, color bg)
  : menu(info.name, bg)
{
  this->selectable = true;
  this->dac = &info;

  ramp_rate_MHz_per_second = 20.0;
  frequency_MHz            = (dac->nyquist_zone==1) ? 1000.0 : 3000.0;
  min_frequency_MHz        = (dac->nyquist_zone==1) ?  100.0 : 2100.0;
  max_frequency_MHz        = (dac->nyquist_zone==1) ? 1900.0 : 3800.0;
  modulate_rate_Mbps       = 100.0;
  amplitude_shift          = 0;

  wgc = hwalloc<white_gauss_control> (HW_RFSoC4x2_BxBDemo1::white_gaussian_noise_0.C_BASEADDR /*0xA0050000*/);
  
  if(info.dac_control)
    {
      printf("Creating a menu_dac_control to control a DAC that already has a hardware address allocated.\n");
      exit(20);
    }
  
  dac->dac_control = hwalloc<swept_spread_phase_generator> (info.hw_address);


  TEXTCOLOR      = scale(info.dac_color, REGIONCOLOR, 1.0);
  SELECTEDCOLOR  = scale(info.dac_color, REGIONCOLOR, 1.0);
  TAGCOLOR       = scale(info.dac_color, REGIONCOLOR, 0.5);

  nyquist_button       = new select_one("Nyquist",   REGIONCOLOR);
  enable_button      = new multiselect("Waveforms", -1, REGIONCOLOR);
  sweep_button       = new select_one("Frequency",   REGIONCOLOR);
  modulate_button    = new select_one("Modulation",  REGIONCOLOR);
  amplitude_button   = new select_one("Sine Amplitude",  REGIONCOLOR);
  gauss_amplitude_button   = new select_one("Noise Level",  REGIONCOLOR);

  nyquist_button->add("Zone 1",  (dac->nyquist_zone==1), false);
  nyquist_button->add("Zone 2",  (dac->nyquist_zone==2), false);

  enable_button->add("Sine", false, false);
  enable_button->add("White Noise", false, false);

  sweep_button->add("Constant",  true, false);
  sweep_button->add("Sweep",    false, false);

  modulate_button->add("None",                   true, false);
  modulate_button->add("BPSK",                   false, false);
  //modulate_button->add("Coninuous Phase BPSK",   false, false); // This isn't even a thing.  My mistake that I could create it.

  amplitude_button->add("  0 dBFS",                    true, false);
  amplitude_button->add(" -6 dBFS",                   false, false);
  amplitude_button->add("-12 dBFS",                   false, false);
  amplitude_button->add("-18 dBFS",                   false, false);
  amplitude_button->add("-24 dBFS",                   false, false);
  amplitude_button->add("-30 dBFS",                   false, false);
  amplitude_button->add("-36 dBFS",                   false, false);
  amplitude_button->add("-42 dBFS",                   false, false);
  amplitude_button->add("-48 dBFS",                   false, false);
  //amplitude_button->add("-54 dBFS",                   false, false);
  //amplitude_button->add("-60 dBFS",                   false, false);

  gauss_amplitude_button->add(" +3 dBFS Max Amp",       true, false);
  gauss_amplitude_button->add("-11 dBFS Gaussian",     false, false);
  gauss_amplitude_button->add("-17 dBFS Gaussian",     false, false);
  gauss_amplitude_button->add("-23 dBFS Gaussian",     false, false);
  gauss_amplitude_button->add("-29 dBFS Gaussian",     false, false);
  gauss_amplitude_button->add("-35 dBFS Gaussian",     false, false);
  gauss_amplitude_button->add("-41 dBFS Gaussian",     false, false);
  gauss_amplitude_button->add("-47 dBFS Gaussian",     false, false);
  //gauss_amplitude_button->add("-53 dBFS Gaussian",     false, false);
  //gauss_amplitude_button->add("-59 dBFS Gaussian",     false, false);
  //gauss_amplitude_button->add("-54 dB",                   false, false);
  //gauss_amplitude_button->add("-60 dB",                   false, false);

  select_trigger = new connection<select_one, selection_changed_event, menu_dac_control, int, int>(this, 0, select_event);
  number_trigger = new connection<number_box, double, menu_dac_control, int, int>(this, 0, number_event);

  nyquist_button   -> add_receiver(select_trigger);
  enable_button    -> add_receiver(select_trigger);
  sweep_button     -> add_receiver(select_trigger);
  modulate_button  -> add_receiver(select_trigger);
  amplitude_button -> add_receiver(select_trigger);
  gauss_amplitude_button -> add_receiver(select_trigger);

  
  color text_color      = scale(WHITE, REGIONCOLOR, 1.0);
  color highlight_color = scale(WHITE, REGIONCOLOR, 0.5);

  float smallest_freq = (dac->nyquist_zone==1) ? 0 : dac_sampling_rate_MHz/2;
  float largest_freq  = (dac->nyquist_zone==1) ? dac_sampling_rate_MHz/2 : dac_sampling_rate_MHz;

  freq_sel = new number_box("Frequency",
			    &frequency_MHz,
			    smallest_freq,
			    largest_freq,
			    dac_freq_list,
			    num_dac_freqs,
			    TYPE_DEFAULT,  // type
			    35, // font_size
			    20, // x_margin
			    20, // y_margin
			    text_color,
			    highlight_color,
			    REGIONCOLOR);

  min_freq_sel = new number_box("Min Frequency",
				&min_frequency_MHz,
				smallest_freq,
				largest_freq,
				dac_freq_list,
				num_dac_freqs,
				TYPE_DEFAULT,  // type
				35, // font_size
				20, // x_margin
				20, // y_margin
				text_color,
				highlight_color,
				REGIONCOLOR);

  max_freq_sel = new number_box("Max Frequency",
				&max_frequency_MHz,
				smallest_freq,
				largest_freq,
				dac_freq_list,
				num_dac_freqs,
				TYPE_DEFAULT,  // type
				35, // font_size
				20, // x_margin
				20, // y_margin
				text_color,
				highlight_color,
				REGIONCOLOR);
  
  ramp_rate_sel = new number_box("Ramp Rate",
				 &ramp_rate_MHz_per_second,
				 0.0,  // min
				 100.01, // max
				 dac_ramp_list,
				 num_dac_ramps,
				 TYPE_DEFAULT,  // type
				 35, // font_size
				 20, // x_margin,
				 20, // y_margin
				 text_color,
				 highlight_color,
				 REGIONCOLOR);

  modulate_rate_sel = new number_box("Modulation Rate",
				     &modulate_rate_Mbps,
				     0.001,
				     200.01,
				     dac_mod_list,
				     num_dac_mods,
				     TYPE_DEFAULT,  // type
				     35, // font_size
				     20, // x_margin,
				     20, // y_margin
				     text_color,
				     highlight_color,
				     REGIONCOLOR);

  freq_sel              -> add_receiver(number_trigger);
  min_freq_sel          -> add_receiver(number_trigger);
  max_freq_sel          -> add_receiver(number_trigger);
  ramp_rate_sel         -> add_receiver(number_trigger);
  modulate_rate_sel     -> add_receiver(number_trigger);

  multiwindow::add(nyquist_button);
  multiwindow::add(enable_button);
  multiwindow::add(sweep_button);
  multiwindow::add(modulate_button);
  multiwindow::add(amplitude_button);
  multiwindow::add(gauss_amplitude_button);

  multiwindow::add(freq_sel);
  multiwindow::add(min_freq_sel);
  multiwindow::add(max_freq_sel);
  multiwindow::add(ramp_rate_sel);
  multiwindow::add(modulate_rate_sel);

  apply_settings();
}


menu_dac_control::~menu_dac_control()
{
  delete select_trigger;
  delete number_trigger;

  wgc->shift_right[dac->dac_number] = 0;

  hwfree(wgc);
  hwfree(dac->dac_control);
  
  multiwindow::remove(nyquist_button);
  multiwindow::remove(enable_button);
  multiwindow::remove(sweep_button);
  multiwindow::remove(modulate_button);
  multiwindow::remove(amplitude_button);
  multiwindow::remove(gauss_amplitude_button);
}


bool menu_dac_control::handle_event(my_event& me)
{
  bool handled = multiwindow::handle_event(me);
  return handled;
}


void menu_dac_control::draw_dirty()
{
  if(!dirty)
    return;
  dirty = false;

  bool sine_enabled  = enable_button->selected[0];
  bool gauss_enabled = enable_button->selected[1];
  //int sweep_num     = sweep_button->get_item_number_selected();
  //int modulate_num  = modulate_button->get_item_number_selected();

  if(nyquist_button->dirty)
    {
      nyquist_button -> draw_dirty();
    }

  if(enable_button->dirty)
    {
      enable_button -> draw_dirty();
    }

  if(sine_enabled)
    {
      if(sweep_button->dirty)
	{
	  sweep_button -> draw_dirty();
	}
      
      if(modulate_button->dirty)
	{
	  modulate_button -> draw_dirty();
	}

      if(amplitude_button->dirty)
	{
	  amplitude_button -> draw_dirty();
	}
    }

  if(gauss_enabled)
    {
      if(gauss_amplitude_button->dirty)
	{
	  gauss_amplitude_button -> draw_dirty();
	}
    }

  // Don't need to worry about number_box items, since they only draw on layout.
}


void menu_dac_control::draw_dynamic()
{
  //enable_button -> draw_dynamic();
  //sweep_button -> draw_dynamic();
  //modulate_button -> draw_dynamic();

  bool sine_enabled  = enable_button->selected[0];
  bool gauss_enabled = enable_button->selected[1];

  int sweep_num     = sweep_button->get_item_number_selected();
  int modulate_num  = modulate_button->get_item_number_selected();

  nyquist_button->draw_dynamic();
  enable_button->draw_dynamic();
  
  if(!sine_enabled)
    {
      // Disabled
    }
  else
    {
      // Enabled

      sweep_button->draw_dynamic();
      modulate_button->draw_dynamic();
      amplitude_button->draw_dynamic();

      if(sweep_num==0)
	{
	  // Constant

	  freq_sel->draw_dynamic();
	}
      else
	{
	  // Sweep
	  min_freq_sel->draw_dynamic();
	  max_freq_sel->draw_dynamic();
	  ramp_rate_sel->draw_dynamic();
	}
      
      if(modulate_num==0)
	{
	  // No modulation
	}
      else if(modulate_num==1)
	{
	  // BPSK modulation

	  modulate_rate_sel->draw_dynamic();
	}
      else
	{
	  // BPSK modulation with continuous phase.  This is the same as FSK modulation with the frequencies being +fs and -fs

	  modulate_rate_sel->draw_dynamic();
	}
    }

  if(gauss_enabled)
    {
      gauss_amplitude_button -> draw_dynamic();
    }

    //multiwindow::draw_dynamic();
}


void menu_dac_control::layout()
{
  float designed_height = 1080.0;
  float designed_width  = 1920.0;
  display* d = get_display();
  float ratio_x = d->width / designed_width;
  float ratio_y = d->height / designed_height;
  float ratio = (ratio_x<ratio_y) ? ratio_x : ratio_y;

  bool sine_enabled  = enable_button->selected[0];
  bool gauss_enabled = enable_button->selected[1];
  int sweep_num     = sweep_button->get_item_number_selected();
  int modulate_num  = modulate_button->get_item_number_selected();
  extern bool ZCU111;
  
  top_margin            = limit(designed_height * 2/100, 5, 13) * ratio;
  bottom_margin         = top_margin;  
  left_margin           = top_margin;
  right_margin          = top_margin;

  mw_margin             = top_margin;

  tag_top_margin        = top_margin/2;
  tag_bottom_margin     = top_margin/2;
  tag_left_margin       = top_margin;
  tag_right_margin      = top_margin;
  
  selection_margin      = top_margin/5;

  tag_left_offset       = left_margin + tag_left_margin;


  nyquist_button_x_offset = 2 * top_margin;
  nyquist_button_y_offset = 2 * top_margin;

  int nyquist_button_desired_height   = (designed_height*ratio - 6*top_margin - bottom_margin)/3;

  nyquist_button->calculate_width_and_height(nyquist_button_width, nyquist_button_height, nyquist_button_desired_height, ratio);

  enable_button_x_offset = 2 * top_margin;
  enable_button_y_offset = nyquist_button_y_offset + nyquist_button_height + 2 * top_margin;
 
  int enable_button_desired_height   = (designed_height*ratio - 6*top_margin - bottom_margin)/3;
  int sweep_button_desired_height    = (designed_height*ratio - 6*top_margin - bottom_margin)/3;
  int modulate_button_desired_height = (designed_height*ratio - 6*top_margin - bottom_margin)/3;


  enable_button->calculate_width_and_height(enable_button_width, enable_button_height, enable_button_desired_height, ratio);


  sweep_button_x_offset = 2 * top_margin;
  sweep_button_y_offset = 2 * top_margin + enable_button_y_offset + enable_button_height;

  sweep_button->calculate_width_and_height(sweep_button_width, sweep_button_height, sweep_button_desired_height, ratio);


  modulate_button_x_offset = 2 * top_margin;
  modulate_button_y_offset = 2 * top_margin + sweep_button_y_offset + sweep_button_height;

  modulate_button->calculate_width_and_height(modulate_button_width, modulate_button_height, modulate_button_desired_height, ratio);

  int button_width = max(max(enable_button_width, sweep_button_width), modulate_button_width);
  int button_height = nyquist_button_height + enable_button_height + 8 * top_margin + sweep_button_height + modulate_button_height;

  int min_freq_sel_desired_height = (designed_height*ratio - 8*top_margin - bottom_margin)/4;
  min_freq_sel_x_offset = 6 * top_margin + button_width;
  min_freq_sel_y_offset = 2 * top_margin;

  // These are essentially identical, but only one is used.
  freq_sel->calculate_width_and_height(min_freq_sel_width, min_freq_sel_height, min_freq_sel_desired_height, ratio);
  min_freq_sel->calculate_width_and_height(min_freq_sel_width, min_freq_sel_height, min_freq_sel_desired_height, ratio);


  int max_freq_sel_desired_height = (designed_height*ratio - 8*top_margin - bottom_margin)/4;
  max_freq_sel_x_offset = 6 * top_margin + button_width;
  max_freq_sel_y_offset = 3 * top_margin + min_freq_sel_height;

  max_freq_sel->calculate_width_and_height(max_freq_sel_width, max_freq_sel_height, max_freq_sel_desired_height, ratio);


  
  int ramp_rate_sel_desired_height = (designed_height*ratio - 8*top_margin - bottom_margin)/4;
  ramp_rate_sel_x_offset = 6 * top_margin + button_width;
  ramp_rate_sel_y_offset = 4 * top_margin + min_freq_sel_height + max_freq_sel_height;

  ramp_rate_sel->calculate_width_and_height(ramp_rate_sel_width, ramp_rate_sel_height, ramp_rate_sel_desired_height, ratio);


  int modulate_rate_sel_desired_height = (designed_height*ratio - 8*top_margin - bottom_margin)/4;
  modulate_rate_sel_x_offset = 6 * top_margin + button_width;

  if(sweep_num==0)
    modulate_rate_sel_y_offset = 3 * top_margin + min_freq_sel_height;
  else
    modulate_rate_sel_y_offset = 5 * top_margin + min_freq_sel_height + max_freq_sel_height + ramp_rate_sel_height;

  modulate_rate_sel->calculate_width_and_height(modulate_rate_sel_width, modulate_rate_sel_height, modulate_rate_sel_desired_height, ratio);


  int enabled_freq_sel_height = (sweep_num==0) ? min_freq_sel_height : min_freq_sel_height + max_freq_sel_height + ramp_rate_sel_height;
  int enabled_modulate_rate_sel_height = (modulate_num==0) ? 0 : modulate_rate_sel_height;
  
  int sel_width = max(max(min_freq_sel_width, max_freq_sel_width), max(ramp_rate_sel_width, modulate_rate_sel_width));
  int sel_height = enabled_freq_sel_height + enabled_modulate_rate_sel_height + 3 * top_margin;


  int amplitude_button_desired_height = (designed_height*ratio - 4*top_margin - bottom_margin)/2;

  amplitude_button_x_offset = modulate_rate_sel_x_offset + modulate_rate_sel_width + 6 * top_margin;
  amplitude_button_y_offset = 2 * top_margin;

  amplitude_button->calculate_width_and_height(amplitude_button_width, amplitude_button_height, amplitude_button_desired_height, ratio);


  int gauss_amplitude_button_desired_height = amplitude_button_desired_height;

  gauss_amplitude_button_x_offset = ( sine_enabled ? modulate_rate_sel_x_offset + modulate_rate_sel_width + 6 * top_margin :
				      /**/           button_width + 2 * top_margin);
  gauss_amplitude_button_y_offset = sine_enabled ? 2 * top_margin + amplitude_button_y_offset + amplitude_button_height : nyquist_button_y_offset;

  gauss_amplitude_button->calculate_width_and_height(gauss_amplitude_button_width, gauss_amplitude_button_height, gauss_amplitude_button_desired_height, ratio);

  //min_freq_sel->layout();
  //max_freq_sel->layout();
  //ramp_rate_sel->layout();
  //modulate_rate_sel->layout();
  

  
  clear(bgcolor);
    
  int circle_width = 20 * ratio;

  //if(top_margin==0) printf("x is zero");
  //if(enable_button_width==0) printf("y is zero");
  //if(ramp_rate_sel_width==0) printf("w is zero");
  //if(bottom_margin==0) printf("h is zero");
  //if(maxheight==0) printf("h is zero");



  /////////////////////////////////////////////////////////////////////////////////////////
  //
  //  Info text
  //
  
  color textcolors[MAX_NUM_DACS+1] = { WHITE, RED, GREEN, BLUE, ORANGE, BLUEGREEN, PURPLE, YELLOW, GREY };

  //
  // Saved as examples of multi colors
  //
  // "Signal Generator Demo\x0B (Forthcoming; now 10MHz sine)",
  // "\x0C" "Full version is for sale.\x0B (Forthcoming)\x0C  Also ask about:",

  int line_height = 40*ratio;
  
  set_text_size(line_height);

  char line1[200] = "";
  char line2[200] = "";
  char line3[200] = "";
  char line4[200] = "";
  char line5[200] = "";
  char line6[200] = "";
  char line7[200] = "";
  char line8[200] = "";
  char line9[200] = "";
  char line10[200] = "";
  char balun_string[100] = "";

  if(dac->dac_number<4)
    sprintf(balun_string, "no balun");
  else if(dac->dac_number<6)
    sprintf(balun_string, "a high-frequency balun");
  else
    sprintf(balun_string, "a low-frequency balun");

  int usual_adc = 7 - dac->dac_number;

  float low_freq_MHz  = (dac->nyquist_zone==1) ? 0.0                 : dac_sampling_rate_MHz/2;
  float high_freq_MHz = (dac->nyquist_zone==1) ? dac_sampling_rate_MHz/2 : dac_sampling_rate_MHz;
  
  sprintf(line1,  "%c%s%c is configured for Nyquist", 11+dac->dac_number, dac_names_hw(dac->dac_number), 10);
  sprintf(line2,  "Zone %d (\"Band %d\", %g to %g MHz).", dac->nyquist_zone, dac->nyquist_zone, low_freq_MHz, high_freq_MHz);

  if(ZCU111)
    {
      sprintf(line3,  "On the XM500, it has %s.", balun_string);
      sprintf(line4,  "Because of the XM500 configuration,");
      sprintf(line5,  "when looped back it is usually");
      sprintf(line6,  "connected to %cADC %d%c.", 11+usual_adc, usual_adc, 10);
      sprintf(line7,  "Looping it back requires DC Blocks.");
      sprintf(line8,  "XM500 ADC/DAC connections are illustrated");
      sprintf(line9,  "in the document \"XM500 Connections\"");
      sprintf(line10, "found in \"Documents\" on the main menu.");
    }
  
  int line1_width  = calculate_text_width(line1);
  int line2_width  = calculate_text_width(line2);
  int line3_width  = calculate_text_width(line3);
  int line4_width  = calculate_text_width(line4);
  int line5_width  = calculate_text_width(line5);
  int line6_width  = calculate_text_width(line6);
  int line7_width  = calculate_text_width(line7);
  int line8_width  = calculate_text_width(line8);
  int line9_width  = calculate_text_width(line9);
  int line10_width = calculate_text_width(line10);

  int line_width = max(max(max(max(max(max(max(max(max(line1_width, line2_width), line3_width), line4_width), line5_width), line6_width), line7_width), line8_width), line9_width), line10_width);

  int rect_width = ( !sine_enabled && !gauss_enabled ? button_width + 2 * top_margin + line_width  :
		     !sine_enabled                   ? gauss_amplitude_button_width + 9*top_margin + enable_button_width  :
		     !gauss_enabled                  ? button_width + 7 * top_margin + sel_width + amplitude_button_width + 4 * top_margin :
		     /**/                              button_width + 7 * top_margin + sel_width + max(amplitude_button_width, gauss_amplitude_button_width)+4*top_margin );

  int explan_text_height = (dac->dac_number<4) ? 16*line_height : 14*line_height;

  int maxheight = ( (!sine_enabled && !gauss_enabled)  ?  explan_text_height              :
		    !sine_enabled                      ?  gauss_amplitude_button_height   :
		    !gauss_enabled                     ?  max(max(button_height, sel_height), amplitude_button_height + 4*top_margin)   :
		    /**/                                  max(max(button_height, sel_height), amplitude_button_height + gauss_amplitude_button_height + 4*top_margin)  );
  int rect_height = maxheight+top_margin+bottom_margin;

  // Shouldn't need this.
  if(rect_height>height-circle_width)
    rect_height = height-circle_width;
  
  //
  //
  /////////////////////////////////////////////////////////////////////////////////////////
  

  draw_rect_with_curved_edges(REGIONCOLOR,
			      left_margin,
			      top_margin,
			      rect_width,
			      rect_height,
			      circle_width);

  nyquist_button->resize(nyquist_button_x_offset, nyquist_button_y_offset, nyquist_button_width, nyquist_button_height);
  nyquist_button->layout();

  enable_button->resize(enable_button_x_offset, enable_button_y_offset, enable_button_width, enable_button_height);
  enable_button->layout();


  int display_width  = get_display()->width;
  int display_height = get_display()->height;
  
  if(!sine_enabled)
    {
      // Disabled

      // Put everything off screen so it can't be clicked on.  Don't lay it out.  No need.
      sweep_button->resize      (display_width, display_height, sweep_button_width, sweep_button_height);
      modulate_button->resize   (display_width, display_height, modulate_button_width, modulate_button_height);
      amplitude_button->resize  (display_width, display_height, amplitude_button_width, amplitude_button_height);
      freq_sel->resize          (display_width, display_height, min_freq_sel_width, min_freq_sel_height);
      min_freq_sel->resize      (display_width, display_height, min_freq_sel_width, min_freq_sel_height);
      max_freq_sel->resize      (display_width, display_height, max_freq_sel_width, max_freq_sel_height);
      ramp_rate_sel->resize     (display_width, display_height, ramp_rate_sel_width, ramp_rate_sel_height);
      modulate_rate_sel->resize (display_width, display_height, modulate_rate_sel_width, modulate_rate_sel_height);

      if(!gauss_enabled)
	{
	  int text_y_offset = nyquist_button_y_offset + 2*top_margin;
	  
	  draw_multicolored_text(line1,
				 textcolors,
				 min_freq_sel_x_offset,
				 text_y_offset,
				 DRAW_TEXT_X_LEFT|DRAW_TEXT_Y_TOP|DRAW_TEXT_ROTATE_0);
	  text_y_offset += line_height * 5/4;
	  
	  draw_multicolored_text(line2,
				 textcolors,
				 min_freq_sel_x_offset,
				 text_y_offset,
				 DRAW_TEXT_X_LEFT|DRAW_TEXT_Y_TOP|DRAW_TEXT_ROTATE_0);
	  text_y_offset += line_height * 5/4;
	  
	  draw_multicolored_text(line3,
				 textcolors,
				 min_freq_sel_x_offset,
				 text_y_offset,
				 DRAW_TEXT_X_LEFT|DRAW_TEXT_Y_TOP|DRAW_TEXT_ROTATE_0);
	  text_y_offset += line_height * 2;
	  
	  draw_multicolored_text(line4,
				 textcolors,
				 min_freq_sel_x_offset,
				 text_y_offset,
				 DRAW_TEXT_X_LEFT|DRAW_TEXT_Y_TOP|DRAW_TEXT_ROTATE_0);
	  text_y_offset += line_height * 5/4;
	  
	  draw_multicolored_text(line5,
				 textcolors,
				 min_freq_sel_x_offset,
				 text_y_offset,
				 DRAW_TEXT_X_LEFT|DRAW_TEXT_Y_TOP|DRAW_TEXT_ROTATE_0);
	  text_y_offset += line_height * 5/4;
	  
	  draw_multicolored_text(line6,
				 textcolors,
				 min_freq_sel_x_offset,
				 text_y_offset,
				 DRAW_TEXT_X_LEFT|DRAW_TEXT_Y_TOP|DRAW_TEXT_ROTATE_0);
	  text_y_offset += line_height * 2;
	  
	  if(dac->dac_number<4)
	    {
	      draw_multicolored_text(line7,
				     textcolors,
				     min_freq_sel_x_offset,
				     text_y_offset,
				     DRAW_TEXT_X_LEFT|DRAW_TEXT_Y_TOP|DRAW_TEXT_ROTATE_0);
	      text_y_offset += line_height * 2;
	    }
	  
	  draw_multicolored_text(line8,
				 textcolors,
				 min_freq_sel_x_offset,
				 text_y_offset,
				 DRAW_TEXT_X_LEFT|DRAW_TEXT_Y_TOP|DRAW_TEXT_ROTATE_0);
	  text_y_offset += line_height * 5/4;
	  
	  draw_multicolored_text(line9,
				 textcolors,
				 min_freq_sel_x_offset,
				 text_y_offset,
				 DRAW_TEXT_X_LEFT|DRAW_TEXT_Y_TOP|DRAW_TEXT_ROTATE_0);
	  text_y_offset += line_height * 5/4;
	  
	  draw_multicolored_text(line10,
				 textcolors,
				 min_freq_sel_x_offset,
				 text_y_offset,
				 DRAW_TEXT_X_LEFT|DRAW_TEXT_Y_TOP|DRAW_TEXT_ROTATE_0);
	  text_y_offset += line_height * 5/4;
	} 
    }
  else
    {
      // Enabled

      sweep_button->resize(sweep_button_x_offset, sweep_button_y_offset, sweep_button_width, sweep_button_height);
      sweep_button->layout();

      modulate_button->resize(modulate_button_x_offset, modulate_button_y_offset, modulate_button_width, modulate_button_height);
      modulate_button->layout();

      amplitude_button->resize(amplitude_button_x_offset, amplitude_button_y_offset, amplitude_button_width, amplitude_button_height);
      amplitude_button->layout();

      if(sweep_num==0)
	{
	  // Constant

	  freq_sel->resize(min_freq_sel_x_offset, min_freq_sel_y_offset, min_freq_sel_width, min_freq_sel_height);
	  freq_sel->layout();

	  // Don't draw min_freq or max_freq or ramp_rate controls.  Put them offscreen so they can't be selected.
	  min_freq_sel->resize      (display_width, display_height, min_freq_sel_width, min_freq_sel_height);
	  max_freq_sel->resize      (display_width, display_height, max_freq_sel_width, max_freq_sel_height);
	  ramp_rate_sel->resize     (display_width, display_height, ramp_rate_sel_width, ramp_rate_sel_height);
	}
      else
	{
	  // Sweep
	  freq_sel->resize          (display_width, display_height, min_freq_sel_width, min_freq_sel_height);

	  min_freq_sel->resize(min_freq_sel_x_offset, min_freq_sel_y_offset, min_freq_sel_width, min_freq_sel_height);
	  min_freq_sel->layout();
  
	  max_freq_sel->resize(max_freq_sel_x_offset, max_freq_sel_y_offset, max_freq_sel_width, max_freq_sel_height);
	  max_freq_sel->layout();

	  ramp_rate_sel->resize(ramp_rate_sel_x_offset, ramp_rate_sel_y_offset, ramp_rate_sel_width, ramp_rate_sel_height);
	  ramp_rate_sel->layout();
	}
      
      if(modulate_num==0)
	{
	  // No modulation

	  modulate_rate_sel->resize (display_width, display_height, modulate_rate_sel_width, modulate_rate_sel_height);
	}
      else if(modulate_num==1)
	{
	  // BPSK modulation

	  modulate_rate_sel->resize(modulate_rate_sel_x_offset, modulate_rate_sel_y_offset, modulate_rate_sel_width, modulate_rate_sel_height);
	  modulate_rate_sel->layout();
	}
      else
	{
	  // BPSK modulation with continuous phase.  This is the same as FSK modulation with the frequencies being +fs and -fs

	  modulate_rate_sel->resize(modulate_rate_sel_x_offset, modulate_rate_sel_y_offset, modulate_rate_sel_width, modulate_rate_sel_height);
	  modulate_rate_sel->layout();
	}
    }

  if(!gauss_enabled)
    {
      // Disabled

      // Put everything off screen so it can't be clicked on.  Don't lay it out.  No need.
      gauss_amplitude_button->resize  (display_width, display_height, amplitude_button_width, amplitude_button_height);
    }
  else
    {
      gauss_amplitude_button->resize(gauss_amplitude_button_x_offset, gauss_amplitude_button_y_offset, gauss_amplitude_button_width, gauss_amplitude_button_height);
      gauss_amplitude_button->layout();
    }
  
  layout_dirty = false;
  dirty = true;
}
