
// SPDX-License-Identifier: CC-BY-NC-SA-4.0
//
// Copyright (C) 2025 Bit by Bit Signal Processing LLC  (https://bxbsp.com)
//
// This work is placed under the "Creative Commons Attribution
// NonCommercial ShareAlike 4.0 International" license, known
// by the shortened acronym "CC-BY-NC-SA-4.0".
//
// This work is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//
// A CC-BY-NC-SA-4.0 license allows you to use, distribute, and modify
// this work, so long as such uses are non-commercial in nature,
// so long as any derived works are offered on the same terms,
// and so long as attribution is given to the original author.
// For further details, see the Creative Commons License
// "CC-BY-NC-SA-4.0".
//
// You should have received a copy of the CC-BY-NC-SA-4.0 license
// along with this work. If not, see
// <https://creativecommons.org/licenses/by-nc-sa/4.0/>.
//


#include <string.h>

#include "menu_graph_select.hh"
#include "data_update.hh"
#include "display.hh"
#include "axis_scale.hh"
#include "are_you_sure.hh"

static int limit(int v, int minval, int maxval)
{
  if(v<minval)
    return minval;
  if(v>maxval)
    return maxval;

  return v;
}

void menu_graph_select::update_x_units_possible_sampling_rate_change()
{
  for(int i=0; i<graph_x_units->num_items; i++)
    {
      if(graph_x_units->selected[i])
	{
	  // Have a selection.  Now find item with matching name

	  for(int j=0; j<num_x_axis_scales; j++)
	    {
	      struct axis_scale* xas = &x_axis_scale_list[j];
	      if(!strcmp(xas->name, graph_x_units->item_name[i]))
		{
		  struct hw_source* s = &hw_source_list[gc.source_list_number[0]];
		  gc.x_data_step   = xas->scale;
		  gc.x_data_start  = s->x_data_start_step_1 * xas->scale;
		  //gc.xmin          = s->xmin_step_1         * xas->scale;  // Handled in menu_config.cc
		  //gc.xmax          = s->xmax_step_1         * xas->scale;
		  sprintf(gc.x_label, "%s", xas->name);
		  break;
		}
	      
	      if(j==num_hw_sources-1)
		{
		  printf("ERROR: selected source doesn't match any from sources list.\n");
		  abort();
		}
	    }
	  
	  break;
	}
    }
}


bool update_graph_due_to_item_change(multiselect*              sender,
				     selection_changed_event   event_data,
				     menu_graph_select*        receiver,
				     int                       receiver_data,
				     int&                      event_return)
{
  // Here change any item except the graph_type

  char title[300];
  char* t = title;
  
  receiver->gc.num_hw_sources = 0;
  graph_type type = receiver->gc.type;


  // Set gc.reference according to selection, adjust title of graph
  if(type==TYPE_Z1_TRANSFER_MAG          ||
     type==TYPE_Z2_TRANSFER_MAG          ||
     type==TYPE_Z1_TRANSFER_PHASE        ||
     type==TYPE_Z2_TRANSFER_PHASE        ||
     type==TYPE_Z1_TRANSFER_GROUP_DELAY  ||
     type==TYPE_Z2_TRANSFER_GROUP_DELAY  ||
     type==TYPE_Z1_TRANSFER_PHASE_DEBUG  ||
     type==TYPE_Z2_TRANSFER_PHASE_DEBUG)
    {
      int selected_ref = receiver->graph_reference->get_item_number_selected();

      graph_type ref_type;
      if(type==TYPE_Z1_TRANSFER_MAG || type==TYPE_Z1_TRANSFER_PHASE || type==TYPE_Z1_TRANSFER_GROUP_DELAY || type==TYPE_Z1_TRANSFER_PHASE_DEBUG )
	ref_type = TYPE_FREQ_Z1;
      else
	ref_type = TYPE_FREQ_Z2;

      int num = 0;
      //int NUM_ADCS = 8; // Hack below to give colors to reference DACs to match ADCs, assuming 8 ADCs, ignoring colors in data_update.cc
      for(int j=0; j<num_hw_sources; j++)
	{
	  struct hw_source* s = &hw_source_list[j];

	  if(s->type==ref_type)
	    {
	      if(num==selected_ref)
		{
		  receiver->gc.reference = s->source_number_sw;
		  t += sprintf(t, "%c%s %cto", 10+graph::MAX_GRAPHS-1, source_names_sw(s->source_number_sw), 10+graph::MAX_GRAPHS);
		  //t += sprintf(t, "%c%s %cto ", 10+(receiver->gc.reference%NUM_ADCS), source_names_sw(s->source_number_sw), 10+graph::MAX_GRAPHS);
		  break;
		}
	      num++;
	    }
	}
    }

  if(type==TYPE_Z1_TRANSFER_PHASE        ||
     type==TYPE_Z2_TRANSFER_PHASE        ||
     type==TYPE_Z1_TRANSFER_GROUP_DELAY  ||
     type==TYPE_Z2_TRANSFER_GROUP_DELAY  ||
     type==TYPE_Z1_TRANSFER_PHASE_DEBUG  ||
     type==TYPE_Z2_TRANSFER_PHASE_DEBUG)
    {
      // Set gc.source_list_number[] according to selection
      for(int i=0; i<receiver->graph_source->num_items; i++)
	{
	  if(receiver->graph_source->selected[i])
	    {
	      // Have a selection.  Now find source with matching name and type
	      
	      for(int j=0; j<num_hw_sources; j++)
		{
		  struct hw_source* s = &hw_source_list[j];
		  if(s->type==type && !strcmp(source_names_sw(s->source_number_sw), receiver->graph_source->item_name[i]))
		    {
		      receiver->gc.source_list_number[receiver->gc.num_hw_sources] = j;
		      t += sprintf(t, " %c%s ", 11+receiver->gc.num_hw_sources, source_names_sw(s->source_number_sw));
		      receiver->gc.num_hw_sources++;
		      break;
		    }
		  
		  if(j==num_hw_sources-1)
		    {
		      printf("ERROR: selected source doesn't match any from sources list.\n");
		      abort();
		    }
		}
	    }
	}
    }
  else
    {
      // Set gc.source_list_number[] according to selection
      for(int i=0; i<receiver->graph_source->num_items; i++)
	{
	  if(receiver->graph_source->selected[i])
	    {
	      // Have a selection.  Now find source with matching name and type
	      
	      for(int j=0; j<num_hw_sources; j++)
		{
		  struct hw_source* s = &hw_source_list[j];
		  if(s->type==type && !strcmp(source_names_sw(s->source_number_sw), receiver->graph_source->item_name[i]))
		    {
		      receiver->gc.source_list_number[receiver->gc.num_hw_sources] = j;
		      t += sprintf(t, " %c%s ", 10+receiver->gc.num_hw_sources, source_names_sw(s->source_number_sw));
		      receiver->gc.num_hw_sources++;
		      break;
		    }
		  
		  if(j==num_hw_sources-1)
		    {
		      printf("ERROR: selected source doesn't match any from sources list.\n");
		      abort();
		    }
		}
	    }
	}
    }

  t += sprintf(t, " %c%s", 10+graph::MAX_GRAPHS, graph_type_names[receiver->gc.type]);

  //printf("Got here.  title is \"%s\"\n", title);
  //   for(int i=0; i<(int)strlen(title); i++)
  //	printf(" %02x", (unsigned char)title[i]);
  //  printf("\n");
  
  sprintf(receiver->gc.title, "%.190s", title);
  
  for(int i=0; i<receiver->graph_x_units->num_items; i++)
    {
      if(receiver->graph_x_units->selected[i])
	{
	  // Have a selection.  Now find item with matching name

	  for(int j=0; j<num_x_axis_scales; j++)
	    {
	      struct axis_scale* s = &x_axis_scale_list[j];
	      if(!strcmp(s->name, receiver->graph_x_units->item_name[i]))
		{
		  double new_scale  = s->scale;
		  double old_scale  = receiver->gc.x_data_step;
		  receiver->gc.x_data_step = new_scale;
		  receiver->gc.xmin *= new_scale / old_scale;
		  receiver->gc.xmax *= new_scale / old_scale;
		  sprintf(receiver->gc.x_label, "%s", s->name);
		  break;
		}
	      
	      if(j==num_hw_sources-1)
		{
		  printf("ERROR: selected source doesn't match any from sources list.\n");
		  abort();
		}
	    }
	  
	  break;
	}
    }

  
  for(int i=0; i<receiver->graph_y_units->num_items; i++)
    {
      if(receiver->graph_y_units->selected[i])
	{
	  // Have a selection.  Now find item with matching name

	  for(int j=0; j<num_y_axis_scales; j++)
	    {
	      struct axis_scale* s = &y_axis_scale_list[j];
	      if(!strcmp(s->name, receiver->graph_y_units->item_name[i]))
		{
		  double new_scale  = s->scale;
		  double old_scale  = receiver->gc.y_scale_factor;
		  receiver->gc.y_scale_factor = new_scale;
		  receiver->gc.ymin *= new_scale / old_scale;
		  receiver->gc.ymax *= new_scale / old_scale;
		  sprintf(receiver->gc.y_label, "%s", s->name);
		  break;
		}
	      
	      if(j==num_hw_sources-1)
		{
		  printf("ERROR: selected source doesn't match any from sources list.\n");
		  abort();
		}
	    }
	  
	  break;
	}
    }

  int averaging_item = receiver->averaging_mode->get_item_number_selected();
  receiver->gc.averaging = ( (averaging_item==0)  ?  AVERAGING_NONE         :
			     (averaging_item==1)  ?  AVERAGING_MINIMAL      :
			     (averaging_item==2)  ?  AVERAGING_MODERATE     :
			     (averaging_item==3)  ?  AVERAGING_LARGE        :
			     (averaging_item==4)  ?  AVERAGING_HUGE         :
			     (averaging_item==5)  ?  AVERAGING_IMMENSE      :
			     (averaging_item==6)  ?  AVERAGING_PEAK_TRACK   :
			     /**/                    AVERAGING_PEAK_HOLD    );

  
  //printf("Graph updated.\n");
  //print_gc(receiver->gc);
  
  return true;
}


// struct graph_config
// {
//   graph_type              type;
//   graph_data_type         data_type;
//   int                     num_hw_addresses;
//   uint32_t                hw_addresses       [graph::MAX_GRAPHS];
//   int                     num_points;
//   double                  x_data_start;
//   double                  x_data_step;
//   double                  y_scale_factor;
//   char                    title              [200];
//   char                    x_label            [200];
//   char                    y_label            [200];
//   double                  xmin;
//   double                  xmax;
//   double                  ymin;
//   double                  ymax;
// };

// struct hw_source
// {
//   graph_type       type;
//   graph_data_type  data_type;
//   char             name[200];
//   uint32_t         hw_address;
//   int              num_points;
//   double           x_data_start_step_1;
//   double           xmin_step_1;
//   double           xmax_step_1;
//   double           ymin_scale_1;
//   double           ymax_scale_1;
// };

bool update_graph_due_to_type_change(multiselect*              sender,
				     selection_changed_event   event_data,
				     menu_graph_select*        receiver,
				     int                       receiver_data,
				     int&                      event_return)
{
  int gt;
  
  for(gt = 0; gt<graph_type::TYPE_LAST; gt++)
    {
      if(!strcmp(graph_type_names[gt], event_data.selection_name))
	{
	  break;
	}

      if(gt+1==graph_type::TYPE_LAST)
	{
	  printf("Error:  selected graph type not on list.\n");
	  abort();
	}
    }

  receiver->rebuild_graph_select_from_gc(sender->get_display(), default_gc((graph_type)gt));
  
  return true;
}



bool delete_graph_action(multiselect*              sender,
			 selection_changed_event   event_data,
			 menu_graph_select*        receiver,
			 int                       receiver_data,
			 int&                      event_return)
{
  menu* graph_menu = (menu*)receiver->parent;

  delete receiver;  // Can't access receiver after this!  :-)

  if(graph_menu->selected_num >= graph_menu->num_subwindows)
    graph_menu->selected_num--;

  if(graph_menu->selected_num == 0)
    graph_menu->selected_num = -1;

  // Renumber any remaining graphs.  Set first to selected, if it exists.
  
  int num = 0;
  // First is the Add action item.  Skip it.
  for(ListElement<window>* le = graph_menu->subwindows[0].first()->next(); le; le=le->next())
    {
      menu* m = (menu*)le->data();
      sprintf(m->tagname, "Graph %d", num++);
    }

  graph_menu->mark_layout_dirty();

  return true;
}


void menu_graph_select::rebuild_graph_select_from_gc(display* disp, const graph_config& gc)
{
  this->gc = gc;

  if(c_graph_type)
    delete c_graph_type;
  if(c_not_graph_type)
    delete c_not_graph_type;
  if(c_graph_delete)
    delete c_graph_delete;

  if(sel_graph_type)
    multiwindow::remove(sel_graph_type);
  if(graph_source)
    multiwindow::remove(graph_source);
  if(graph_reference)
    multiwindow::remove(graph_reference);
  if(graph_x_units)
    multiwindow::remove(graph_x_units);
  if(graph_y_units)
    multiwindow::remove(graph_y_units);
  if(averaging_mode)
    multiwindow::remove(averaging_mode);
  if(delete_graph)
    multiwindow::remove(delete_graph);
  
  sel_graph_type = new select_one("Graph Type", REGIONCOLOR);

  for(int gt = 0; gt<graph_type::TYPE_LAST; gt++)
    {
      // See if there is a source for this type
      bool has_source = false;
      for(int i=0; i<num_hw_sources; i++)
	{
	  struct hw_source* s = &hw_source_list[i];
	  if(s->type==gt)
	      {
		has_source = true;
		break;
	      }
	}

      // printf("Adding \"%s\".\n", graph_type_names[gt]);
      
      sel_graph_type->add(graph_type_names[gt], (gc.type==gt), !has_source);
    }

  graph_reference = new select_one("From", REGIONCOLOR);
      
  if(gc.type==TYPE_Z1_TRANSFER_MAG          ||
     gc.type==TYPE_Z2_TRANSFER_MAG          ||
     gc.type==TYPE_Z1_TRANSFER_PHASE        ||
     gc.type==TYPE_Z2_TRANSFER_PHASE        ||
     gc.type==TYPE_Z1_TRANSFER_GROUP_DELAY  ||
     gc.type==TYPE_Z2_TRANSFER_GROUP_DELAY  ||
     gc.type==TYPE_Z1_TRANSFER_PHASE_DEBUG  ||
     gc.type==TYPE_Z2_TRANSFER_PHASE_DEBUG  )
    {
      graph_type ref_type;

      if(gc.type==TYPE_Z1_TRANSFER_MAG || gc.type==TYPE_Z1_TRANSFER_PHASE || gc.type==TYPE_Z1_TRANSFER_GROUP_DELAY || gc.type==TYPE_Z1_TRANSFER_PHASE_DEBUG)
	ref_type = TYPE_FREQ_Z1;
      else
	ref_type = TYPE_FREQ_Z2;

      int reference_num = 0;
      for(int i=0; i<num_hw_sources; i++)
	{
	  struct hw_source* s = &hw_source_list[i];
	  if(s->type==ref_type)
	    {
	      bool selected = (gc.reference==s->source_number_sw);
	      graph_reference->add(source_names_sw(s->source_number_sw), selected, false);
	      color text_color = scale(source_colors_sw(s->source_number_sw), REGIONCOLOR, 1.0);
	      graph_reference->set_text_color(reference_num, text_color);
	      graph_reference->set_selected_text_color(reference_num, source_colors_sw(s->source_number_sw));
	      color highlight_color = scale(source_colors_sw(s->source_number_sw), REGIONCOLOR, 0.5);
	      graph_reference->set_selected_highlight_color(reference_num, highlight_color);
	      reference_num++;
	    }
	}

      if( gc.type==TYPE_Z1_TRANSFER_MAG || gc.type==TYPE_Z2_TRANSFER_MAG )
	graph_source = new multiselect("To", 4, REGIONCOLOR);
      else
	graph_source = new select_one("To", REGIONCOLOR);
    }
  else
    {
      graph_source = new multiselect("Source(s)", 4, REGIONCOLOR);
    }

  int source_num = 0;
  for(int i=0; i<num_hw_sources; i++)
    {
      struct hw_source* s = &hw_source_list[i];
      if(s->type==gc.type)
	{
	  bool selected = false;
	  for(int j=0; j<gc.num_hw_sources; j++)
	    if(gc.source_list_number[j]==i)
	      {
		selected=true;
		break;
	      }

	  graph_source->add(source_names_sw(s->source_number_sw), selected, false);
	  color text_color = scale(source_colors_sw(s->source_number_sw), REGIONCOLOR, 1.0);
	  graph_source->set_text_color(source_num, text_color);
	  graph_source->set_selected_text_color(source_num, source_colors_sw(s->source_number_sw));
	  color highlight_color = scale(source_colors_sw(s->source_number_sw), REGIONCOLOR, 0.5);
	  graph_source->set_selected_highlight_color(source_num, highlight_color);
	  source_num++;
	}
    }

  graph_x_units = new select_one("X Units", REGIONCOLOR);


  bool x_scale_matches = false;
  for(int i=0; i<num_x_axis_scales; i++)
    {
      struct axis_scale* xas = &x_axis_scale_list[i];
      if(xas->type==gc.type)
	{
	  if(gc.x_data_step==xas->scale)
	    {
	      x_scale_matches = true;
	      break;
	    }
	}
    }
  
  for(int i=0; i<num_x_axis_scales; i++)
    {
      struct axis_scale* xas = &x_axis_scale_list[i];
      if(xas->type==gc.type)
	{
	  bool selected = x_scale_matches ? (gc.x_data_step==xas->scale) : xas->is_default;
	  graph_x_units->add(xas->name, selected, false);
	  //if(xas->is_default)
	  //  gc.x_data_step = xas->scale;
	}
    }

  graph_y_units = new select_one("Y Units", REGIONCOLOR);

  bool y_scale_matches = false;
  for(int i=0; i<num_y_axis_scales; i++)
    {
      struct axis_scale* yas = &y_axis_scale_list[i];
      if(yas->type==gc.type)
	{
	  if(gc.y_scale_factor==yas->scale)
	    {
	      y_scale_matches = true;
	      break;
	    }
	}
    }

  for(int i=0; i<num_y_axis_scales; i++)
    {
      struct axis_scale* yas = &y_axis_scale_list[i];
      if(yas->type==gc.type)
	{
	  bool selected = y_scale_matches ? (gc.y_scale_factor==yas->scale) : yas->is_default;
	  graph_y_units->add(yas->name, selected, false);
	  //if(yas->is_default)
	  //  gc.y_scale_factor = yas->scale;
	}
    }

  averaging_mode = new select_one("Averaging", REGIONCOLOR);
  averaging_mode->add("None",          (gc.averaging==AVERAGING_NONE),       false);
  averaging_mode->add("Minimal",       (gc.averaging==AVERAGING_MINIMAL),    false);
  averaging_mode->add("Moderate",      (gc.averaging==AVERAGING_MODERATE),   false);
  averaging_mode->add("Large",         (gc.averaging==AVERAGING_LARGE),      false);
  averaging_mode->add("Huge",          (gc.averaging==AVERAGING_HUGE),       false);
  averaging_mode->add("Immense",       (gc.averaging==AVERAGING_IMMENSE),    false);

  if(gc.type==TYPE_FREQ_Z1 || gc.type==TYPE_FREQ_Z2)
    {
      averaging_mode->add("Brief Peak Hold",    (gc.averaging==AVERAGING_PEAK_TRACK),  false);
      averaging_mode->add("Infinite Peak Hold",     (gc.averaging==AVERAGING_PEAK_HOLD),   false);
    }

  delete_graph = new button("Delete",        // button_name
			    20,              // font_size
			    10,              // x_margin
			    10,              // y_margin
			    RED,             // text color
			    REDRED6,         // fg color
			    REGIONCOLOR);    // bg color

  multiwindow::add(sel_graph_type);
  multiwindow::add(graph_source);
  multiwindow::add(graph_reference);
  multiwindow::add(graph_x_units);
  multiwindow::add(graph_y_units);
  multiwindow::add(averaging_mode);
  multiwindow::add(delete_graph);

  c_graph_type      = new connection<multiselect, selection_changed_event, menu_graph_select, int, int>(this, 0, update_graph_due_to_type_change);
  c_not_graph_type  = new connection<multiselect, selection_changed_event, menu_graph_select, int, int>(this, 0, update_graph_due_to_item_change);
  c_graph_delete    = new connection<multiselect, selection_changed_event, menu_graph_select, int, int>(this, 0, delete_graph_action);

  sel_graph_type   -> add_receiver(c_graph_type);
  graph_source     -> add_receiver(c_not_graph_type);
  graph_reference  -> add_receiver(c_not_graph_type);
  graph_x_units    -> add_receiver(c_not_graph_type);
  graph_y_units    -> add_receiver(c_not_graph_type);
  averaging_mode   -> add_receiver(c_not_graph_type);
  delete_graph     -> add_receiver(c_graph_delete);

  if(graph_source->get_num_selected()==0)
    {
      disp->claim_events(this);
    }
}



void menu_graph_select::this_selected()
{
  //printf("Graph Menu %p selected.\n", this);
  mark_layout_dirty();
  parent->mark_layout_dirty();
}

menu_graph_select::menu_graph_select(display* disp, const char* tagname, const graph_config& gc, color bg)
  : menu(tagname, bg)
{
  this->selectable = true;

  sel_graph_type = 0;
  graph_source = 0;
  graph_reference = 0;
  graph_x_units = 0;
  graph_y_units = 0;
  averaging_mode = 0;
  delete_graph = 0;
  
  c_graph_type = 0;
  c_not_graph_type = 0;
  c_graph_delete = 0;

  rebuild_graph_select_from_gc(disp, gc);
}


menu_graph_select::~menu_graph_select()
{
  if(c_graph_type)
    delete c_graph_type;
  if(c_not_graph_type)
    delete c_not_graph_type;

  if(sel_graph_type)
    multiwindow::remove(sel_graph_type);
  if(graph_source)
    multiwindow::remove(graph_source);
  if(graph_reference)
    multiwindow::remove(graph_reference);
  if(graph_x_units)
    multiwindow::remove(graph_x_units);
  if(graph_y_units)
    multiwindow::remove(graph_y_units);
  if(averaging_mode)
    multiwindow::remove(averaging_mode);

  //printf("deleting menu_graph_select\n");
}


bool menu_graph_select::handle_event(my_event& me)
{
  display* disp = get_display();

  //
  // This ends up handling the delete_graph button twice, but
  // that doesn't hurt.  The reason for doing this is that if
  // we are deleted, we want to exit immediately because
  // any actions we take on a deleted class can result in a
  // core dump.
  //
  bool deleted = delete_graph->handle_event(me);
  
  if(deleted)
    return true;
  
  bool handled = multiwindow::handle_event(me);
  
  if(graph_source->get_num_selected()==0)
    {
      claim_events();

      if(!handled && (me.type == EVENT_TOUCH))
	{
	  // Nothing that was selected fixed the no-sources problem, or changed anything else
	  disp->entry_rejected_beep();

	  int xoff = 0;
	  int yoff = 0;
	  window* p = this;
	  while(p)
	    {
	      xoff += p->offset.x;
	      yoff += p->offset.y;
	      p = p->parent;
	    }
	  
	  if(me.c[0].x >= xoff + rect_left    &&
	     me.c[0].x <  xoff + rect_right   && 
	     me.c[0].y >= yoff + rect_top     && 
	     me.c[0].y <  yoff + rect_bottom  )
	    {
	      sel_graph_type   -> highlight_count = 50;
	      graph_source     -> highlight_count = 100;
	      graph_reference  -> highlight_count = 50;
	      graph_x_units    -> highlight_count = 50;
	      graph_y_units    -> highlight_count = 50;
	      averaging_mode   -> highlight_count = 50;
	      delete_graph     -> highlight_count = 100;
	    }
	  else
	    {
	      ask_are_you_sure(get_display(),
			       "Please select sources to graph, or delete the graph.",
			       0,
			       "OK",
			       0,
			       0,
			       0,
			       0);
	    }

	}
      return true;
    }
  else
    {
      release_events();
    }

  return handled;
}


void menu_graph_select::draw_dirty()
{
  //printf("dirty win x=%d, y=%d, (%d,%d)\n", offset.x, offset.y, width, height);
  //multiwindow* mw = parent;
  //while(mw)
  //  {
  //    printf("   parent x=%d, y=%d, (%d,%d)\n", mw->offset.x, mw->offset.y, mw->width, mw->height);
  //    mw = mw->parent;
  //  }

  if(!dirty)
    return;
  dirty = false;

  int type = sel_graph_type->get_item_number_selected();
   
  if(sel_graph_type->dirty)
    sel_graph_type  -> draw_dirty();

  if(graph_source->dirty)
    graph_source    -> draw_dirty();

  if(graph_reference->dirty)
    graph_reference -> draw_dirty();

  if(graph_x_units->dirty)
    graph_x_units   -> draw_dirty();

  if(graph_y_units->dirty)
    graph_y_units   -> draw_dirty();

  if(averaging_mode->dirty && (type==TYPE_FREQ_Z1 || type==TYPE_FREQ_Z2 || type==TYPE_Z1_TRANSFER_MAG || type==TYPE_Z1_TRANSFER_PHASE_DEBUG || type==TYPE_Z1_TRANSFER_PHASE || type==TYPE_Z2_TRANSFER_MAG  || type==TYPE_Z2_TRANSFER_PHASE_DEBUG || type==TYPE_Z2_TRANSFER_PHASE || type==TYPE_Z1_TRANSFER_GROUP_DELAY || type==TYPE_Z2_TRANSFER_GROUP_DELAY ))
    averaging_mode  -> draw_dirty();

  if(delete_graph->dirty)
    delete_graph    -> draw_dirty();

  //printf("dirty end.\n");
}


void menu_graph_select::draw_dynamic()
{
  int type = sel_graph_type->get_item_number_selected();
  
  sel_graph_type  -> draw_dynamic();
  graph_source    -> draw_dynamic();
  graph_reference -> draw_dynamic();
  graph_x_units   -> draw_dynamic();
  graph_y_units   -> draw_dynamic();
  
  if(type==TYPE_FREQ_Z1 || type==TYPE_FREQ_Z2 || type==TYPE_Z1_TRANSFER_MAG || type==TYPE_Z1_TRANSFER_PHASE_DEBUG || type==TYPE_Z1_TRANSFER_PHASE || type==TYPE_Z2_TRANSFER_MAG  || type==TYPE_Z2_TRANSFER_PHASE_DEBUG || type==TYPE_Z2_TRANSFER_PHASE || type==TYPE_Z1_TRANSFER_GROUP_DELAY || type==TYPE_Z2_TRANSFER_GROUP_DELAY )
    averaging_mode  -> draw_dynamic();

  delete_graph    -> draw_dynamic();
}


void menu_graph_select::layout()
{  
  //printf("in menu_graph_select::layout(), laying out \"%s\".\n", tagname);
  
  //
  //printf("win x=%d, y=%d, (%d,%d)\n", offset.x, offset.y, width, height);
  //multiwindow* mw = parent;
  //while(mw)
  //  {
  //    printf("   parent x=%d, y=%d, (%d,%d)\n", mw->offset.x, mw->offset.y, mw->width, mw->height);
  //    mw = mw->parent;
  //  }
  //
  //printf("in menu_graph_select::layout() 2;\n");

  float designed_height = 1080.0;
  float designed_width  = 1920.0;
  display* d = get_display();
  float ratio_x = d->width / designed_width;
  float ratio_y = d->height / designed_height;
  float ratio = (ratio_x<ratio_y) ? ratio_x : ratio_y;

  top_margin            = limit(designed_height * 2/100, 5, 13) * ratio;
  bottom_margin         = top_margin;  
  left_margin           = top_margin;
  right_margin          = top_margin;

  mw_margin             = top_margin;

  delete_margin         = 2*top_margin;
  
  tag_top_margin        = top_margin/2;
  tag_bottom_margin     = top_margin/2;
  tag_left_margin       = top_margin;
  tag_right_margin      = top_margin;
  
  selection_margin      = top_margin/5;

  tag_left_offset       = left_margin + tag_left_margin;

  int graph_type_x_offset = 2 * top_margin;
  int graph_type_y_offset = 2 * top_margin;

  int graph_type_width;
  int graph_type_height;
  int graph_type_desired_height = designed_height*ratio - top_margin - bottom_margin;

  sel_graph_type->calculate_width_and_height(graph_type_width, graph_type_height, graph_type_desired_height, ratio);


  int graph_reference_x_offset = graph_type_x_offset + 2 * top_margin + graph_type_width;
  int graph_reference_y_offset = 2 * top_margin;

  int graph_reference_width;
  int graph_reference_height;
  int graph_reference_desired_height = designed_height*ratio - top_margin - bottom_margin;

  graph_reference->calculate_width_and_height(graph_reference_width, graph_reference_height, graph_reference_desired_height, ratio);

  int graph_source_x_offset;

  if(gc.type==TYPE_Z1_TRANSFER_MAG          ||
     gc.type==TYPE_Z2_TRANSFER_MAG          ||
     gc.type==TYPE_Z1_TRANSFER_PHASE        ||
     gc.type==TYPE_Z2_TRANSFER_PHASE        ||
     gc.type==TYPE_Z1_TRANSFER_GROUP_DELAY  ||
     gc.type==TYPE_Z2_TRANSFER_GROUP_DELAY  ||
     gc.type==TYPE_Z1_TRANSFER_PHASE_DEBUG  ||
     gc.type==TYPE_Z2_TRANSFER_PHASE_DEBUG  )
    {
      graph_source_x_offset = graph_reference_x_offset + 2 * top_margin + graph_reference_width;
    }
  else
    {
      graph_source_x_offset = graph_type_x_offset + 2 * top_margin + graph_type_width;
    }
      
  int graph_source_y_offset = 2 * top_margin;

  int graph_source_width;
  int graph_source_height;
  int graph_source_desired_height = designed_height*ratio - top_margin - bottom_margin;

  graph_source->calculate_width_and_height(graph_source_width, graph_source_height, graph_source_desired_height, ratio);


  int graph_x_units_x_offset = graph_source_x_offset + 2 * top_margin + graph_source_width;
  int graph_x_units_y_offset = 2 * top_margin;

  int graph_x_units_width;
  int graph_x_units_height;
  int graph_x_units_desired_height = designed_height*ratio - top_margin - bottom_margin;

  graph_x_units->calculate_width_and_height(graph_x_units_width, graph_x_units_height, graph_x_units_desired_height, ratio);


  int graph_y_units_x_offset = graph_x_units_x_offset + 2 * top_margin + graph_x_units_width;
  int graph_y_units_y_offset = 2 * top_margin;

  int graph_y_units_width;
  int graph_y_units_height;
  int graph_y_units_desired_height = designed_height*ratio*2/3 - top_margin - bottom_margin;

  graph_y_units->calculate_width_and_height(graph_y_units_width, graph_y_units_height, graph_y_units_desired_height, ratio);


  int averaging_mode_x_offset = graph_x_units_x_offset + 2 * top_margin + graph_x_units_width;
  int averaging_mode_y_offset = 4 * top_margin + graph_y_units_height;

  int averaging_mode_width;
  int averaging_mode_height;
  int averaging_mode_desired_height = designed_height*ratio/3 - top_margin - bottom_margin;

  averaging_mode->calculate_width_and_height(averaging_mode_width, averaging_mode_height, averaging_mode_desired_height, ratio);

  int right_side = averaging_mode_x_offset + max(averaging_mode_width, graph_y_units_width);

  int delete_graph_width;
  int delete_graph_height;
  int delete_graph_desired_height = designed_height*ratio - top_margin - bottom_margin;

  delete_graph->font_height = sel_graph_type->item_font_height;

  delete_graph->calculate_width_and_height(delete_graph_width, delete_graph_height, delete_graph_desired_height, ratio);

  
  int maxheight = max(max(max(max(graph_type_height+delete_graph_height+2*delete_margin, graph_reference_height), graph_source_height), graph_x_units_height), graph_y_units_height+averaging_mode_height + 2 * top_margin);;
  
  clear(bgcolor);
    
  int circle_width = 20;

  rect_left    = left_margin;
  rect_right   = right_side - left_margin + top_margin;
  rect_top     = top_margin;
  rect_bottom  = maxheight+top_margin+bottom_margin;

  draw_rect_with_curved_edges(REGIONCOLOR, rect_left, rect_top, rect_right, rect_bottom, circle_width);


  sel_graph_type->resize(graph_type_x_offset, graph_type_y_offset, graph_type_width, graph_type_height);
  sel_graph_type->layout();
  graph_source->resize(graph_source_x_offset, graph_source_y_offset, graph_source_width, graph_source_height);
  graph_source->layout();
  graph_x_units->resize(graph_x_units_x_offset, graph_x_units_y_offset, graph_x_units_width, graph_x_units_height);
  graph_x_units->layout();
  graph_y_units->resize(graph_y_units_x_offset, graph_y_units_y_offset, graph_y_units_width, graph_y_units_height);
  graph_y_units->layout();

  int type = gc.type;//sel_graph_type->get_item_number_selected();

  if(type==TYPE_FREQ_Z1 || type==TYPE_FREQ_Z2 || type==TYPE_Z1_TRANSFER_MAG || type==TYPE_Z1_TRANSFER_PHASE_DEBUG || type==TYPE_Z1_TRANSFER_PHASE || type==TYPE_Z2_TRANSFER_MAG  || type==TYPE_Z2_TRANSFER_PHASE_DEBUG || type==TYPE_Z2_TRANSFER_PHASE  || type==TYPE_Z1_TRANSFER_GROUP_DELAY || type==TYPE_Z2_TRANSFER_GROUP_DELAY)
    {
      averaging_mode->resize(averaging_mode_x_offset, averaging_mode_y_offset, averaging_mode_width, averaging_mode_height);
      averaging_mode->layout();
    }
  else
    {
      // Put off-screen so it can't be clicked on
      display* d = get_display();
      averaging_mode->resize(d->width, d->height, averaging_mode_width, averaging_mode_height);
    }

  if(gc.type==TYPE_Z1_TRANSFER_MAG          ||
     gc.type==TYPE_Z2_TRANSFER_MAG          ||
     gc.type==TYPE_Z1_TRANSFER_PHASE        ||
     gc.type==TYPE_Z2_TRANSFER_PHASE        ||
     gc.type==TYPE_Z1_TRANSFER_GROUP_DELAY  ||
     gc.type==TYPE_Z2_TRANSFER_GROUP_DELAY  ||
     gc.type==TYPE_Z1_TRANSFER_PHASE_DEBUG  ||
     gc.type==TYPE_Z2_TRANSFER_PHASE_DEBUG  )
    {
      graph_reference->resize(graph_reference_x_offset, graph_reference_y_offset, graph_reference_width, graph_reference_height);
      graph_reference->layout();
    }
  else
    {
      // Put off-screen so it can't be clicked on
      display* d = get_display();
      graph_reference->resize(d->width, d->height, graph_reference_width, graph_reference_height);
    }
    
  delete_graph->resize(graph_type_x_offset + (graph_type_width-delete_graph_width)/2, graph_type_y_offset + graph_type_height + delete_margin, delete_graph_width, delete_graph_height);
  delete_graph->layout();

  layout_dirty = false;
  dirty = true;
}
