
// SPDX-License-Identifier: CC-BY-NC-SA-4.0
//
// Copyright (C) 2025 Bit by Bit Signal Processing LLC  (https://bxbsp.com)
//
// This work is placed under the "Creative Commons Attribution
// NonCommercial ShareAlike 4.0 International" license, known
// by the shortened acronym "CC-BY-NC-SA-4.0".
//
// This work is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//
// A CC-BY-NC-SA-4.0 license allows you to use, distribute, and modify
// this work, so long as such uses are non-commercial in nature,
// so long as any derived works are offered on the same terms,
// and so long as attribution is given to the original author.
// For further details, see the Creative Commons License
// "CC-BY-NC-SA-4.0".
//
// You should have received a copy of the CC-BY-NC-SA-4.0 license
// along with this work. If not, see
// <https://creativecommons.org/licenses/by-nc-sa/4.0/>.
//


#include <string.h>

#include "menu_trigger_select.hh"
#include "data_update.hh"
#include "display.hh"

int     menu_trigger_select::trigger_source      =    -1;
int     menu_trigger_select::num_sources         =     0;
double  menu_trigger_select::trigger_level       =   0.0;
double  menu_trigger_select::trigger_hysteresis  = 200.0;

static int limit(int v, int minval, int maxval)
{
  if(v<minval)
    return minval;
  if(v>maxval)
    return maxval;

  return v;
}


// struct hw_source
// {
//   graph_type       type;
//   graph_data_type  data_type;
//   char             name[200];
//   uint32_t         hw_address;
//   int              num_points;
//   double           x_data_start_step_1;
//   double           xmin_step_1;
//   double           xmax_step_1;
//   double           ymin_scale_1;
//   double           ymax_scale_1;
// };

bool source_select(select_one*               sender,
		   selection_changed_event   event_data,
		   menu_trigger_select*      receiver,
		   int                       receiver_data,
		   int&                      event_return)
{
  receiver->trigger_source = event_data.selection_number;

  // Last source, source number num_sources-1, is Free Run, which is no source at all.
  if(receiver->trigger_source>=receiver->num_sources-1 || receiver->trigger_source<0)
    receiver->trigger_source = -1;
  event_return = 0;
  return true;
}


void menu_trigger_select::this_selected()
{
  //printf("Graph Menu %p selected.\n", this);
  mark_layout_dirty();
  parent->mark_layout_dirty();
}



menu_trigger_select::menu_trigger_select(const char* tagname, color bg)
  : menu(tagname, bg)
{
  this->selectable = true;

  trigger_source_sel = new select_one("Trigger Source", REGIONCOLOR);

  num_sources = 0;
  for(int i=0; i<num_hw_sources; i++)
    {
      struct hw_source* s = &hw_source_list[i];
      if(s->type==TYPE_TIME)
	{
	  bool selected = (trigger_source==num_sources);

	  trigger_source_sel->add(source_names_sw(s->source_number_sw), selected, false);
	  color text_color      = scale(source_colors_sw(s->source_number_sw), REGIONCOLOR, 1.0);
	  color highlight_color = scale(source_colors_sw(s->source_number_sw), REGIONCOLOR, 0.5);
	  trigger_source_sel->set_text_color               (num_sources, text_color);
	  trigger_source_sel->set_selected_text_color      (num_sources, source_colors_sw(s->source_number_sw));
	  trigger_source_sel->set_selected_highlight_color (num_sources, highlight_color);
	  num_sources++;
	}
    }


  if(trigger_source<0 || trigger_source>num_sources)
    trigger_source = -1;

  trigger_source_sel->add("Free Run", trigger_source==-1, false);
  color free_run_color           = BLACK;
  color free_run_text_color      = scale(free_run_color, WHITE, 1.0);
  color free_run_highlight_color = scale(free_run_color, WHITE, 0.5);
  trigger_source_sel->set_text_color               (num_sources, free_run_text_color);
  trigger_source_sel->set_selected_text_color      (num_sources, free_run_color);
  trigger_source_sel->set_selected_highlight_color (num_sources, free_run_highlight_color);
  num_sources++;


  c_trigger = new connection<select_one, selection_changed_event, menu_trigger_select, int, int>(this, 0, source_select);


  trigger_source_sel -> add_receiver(c_trigger);

  color text_color      = scale(WHITE, REGIONCOLOR, 1.0);
  color highlight_color = scale(WHITE, REGIONCOLOR, 0.5);

  trigger_level_sel = new number_box("Trigger Level",
				     &trigger_level,
				     -32768,
				     32767,
				     y_axis_scale_list,
				     num_y_axis_scales,
				     TYPE_TIME,
				     35, // font_size
				     20, // x_margin,
				     20, // y_margin
				     text_color,
				     highlight_color,
				     REGIONCOLOR);


  multiwindow::add(trigger_source_sel);
  multiwindow::add(trigger_level_sel);
}


menu_trigger_select::~menu_trigger_select()
{
  delete c_trigger;

  multiwindow::remove(trigger_source_sel);
}


bool menu_trigger_select::handle_event(my_event& me)
{
  bool handled = multiwindow::handle_event(me);
  return handled;
}


void menu_trigger_select::draw_dirty()
{
  if(!dirty)
    return;
  dirty = false;
  
  if(trigger_source_sel->dirty)
    trigger_source_sel -> draw_dirty();
}


void menu_trigger_select::draw_dynamic()
{
  //trigger_source_sel -> draw_dynamic();
  multiwindow::draw_dynamic();
}


void menu_trigger_select::layout()
{
  float designed_height = 1080.0;
  float designed_width  = 1920.0;
  display* d = get_display();
  float ratio_x = d->width / designed_width;
  float ratio_y = d->height / designed_height;
  float ratio = (ratio_x<ratio_y) ? ratio_x : ratio_y;
  
  top_margin            = limit(designed_height * 2/100, 5, 13) * ratio;
  bottom_margin         = top_margin;  
  left_margin           = top_margin;
  right_margin          = top_margin;

  mw_margin             = top_margin;

  tag_top_margin        = top_margin/2;
  tag_bottom_margin     = top_margin/2;
  tag_left_margin       = top_margin;
  tag_right_margin      = top_margin;
  
  selection_margin      = top_margin/5;

  tag_left_offset       = left_margin + tag_left_margin;


  trigger_source_sel_x_offset = 2 * top_margin;
  trigger_source_sel_y_offset = 2 * top_margin;

 
  int trigger_source_sel_desired_height = designed_height*ratio - top_margin - bottom_margin;
  int trigger_level_sel_desired_height = designed_height*ratio - top_margin - bottom_margin;

  trigger_source_sel->calculate_width_and_height(trigger_source_sel_width, trigger_source_sel_height, trigger_source_sel_desired_height, ratio);

  trigger_level_sel_x_offset = 3 * top_margin + trigger_source_sel_width;
  trigger_level_sel_y_offset = 2 * top_margin;

  trigger_level_sel->calculate_width_and_height(trigger_level_sel_width, trigger_level_sel_height, trigger_level_sel_desired_height, ratio);

  trigger_level_sel->layout();
  


  int maxheight = max(trigger_source_sel_height, trigger_level_sel_height);
  
  clear(bgcolor);
    
  int circle_width = 20 * ratio;

  if(top_margin==0) printf("x is zero");
  if(trigger_source_sel_width==0) printf("y is zero");
  if(trigger_level_sel_width==0) printf("w is zero");
  if(bottom_margin==0) printf("h is zero");
  if(maxheight==0) printf("h is zero");

  
  draw_rect_with_curved_edges(REGIONCOLOR,
			      left_margin,
			      top_margin,
			      trigger_source_sel_width + 3 * top_margin + trigger_level_sel_width,
			      maxheight+top_margin+bottom_margin,
			      circle_width);


  trigger_source_sel->resize(trigger_source_sel_x_offset, trigger_source_sel_y_offset, trigger_source_sel_width, trigger_source_sel_height);
  trigger_source_sel->layout();

  trigger_level_sel->resize(trigger_level_sel_x_offset, trigger_level_sel_y_offset, trigger_level_sel_width, trigger_level_sel_height);
  trigger_level_sel->layout();
      
  layout_dirty = false;
  dirty = true;
}
