
// SPDX-License-Identifier: CC-BY-NC-ND-4.0
//
// Copyright (C) 2025 Bit by Bit Signal Processing LLC (https://bxbsp.com)
//
// This work is placed under the "Creative Commons Attribution
// NonCommercial NoDerivatives 4.0 International" license, known
// by the shortened acronym "CC-BY-NC-ND-4.0".
//
// This work is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//
// A CC-BY-NC-ND-4.0 license allows you to use this work for
// noncommercial purposes so long as attribution is made to the
// original author.  This work may be distributed in unmodified form,
// but derivatives of this work may not be distributed.  For further
// details, see the Creative Commons License "CC-BY-NC-ND-4.0".
//
// You should have received a copy of the CC-BY-NC-ND-4.0 license
// along with this work. If not, see
// <https://creativecommons.org/licenses/by-nc-nd/4.0/>.
//


#ifndef BASE_DATA_HH
#define BASE_DATA_HH

#include <math.h>

//#include "graph.hh"

//
// This is a base class that holds data and allows it to be updated by means
// specified in derived classes.
//

template <typename T> class base_data
{
  //private:
public:
  bool delete_data;

 public:
  bool direct_access;
  int  num_points;

  // direct access data may be contiguous, in which case group_length=1.
  // If non-contiguous, say for a PPC=10 time plot, data is in groups of
  // 16 with the first 10 valid.  So group_length=16 and
  // valid_per_group=10.
  int  group_length;
  int  valid_per_group;
  
  class graph* g;  // Pointer back to graph using this data.  null if we don't know

  // May read data directly if direct_access is true, for efficiency.
  T*   data;

  // Access read this way any time; read only through this function if direct_access is false
  virtual T get_data (int index);

  // Scale is set by the end user to say what is expected regarding the x and y of the data.
  // This may be used to create the dataset, if these are set top down.  Often these are
  // set bottom up, so that the top level knows the scale of the data already and this
  // function does nothing.
  virtual void set_scale(double x_data_start, double x_data_step, double y_scale);

  virtual void update_data()   {}   // Incremental updates, not tied to display  (e.g. averaging)
  virtual void reset_data()    {}   // Reset any updates
  virtual void finalize_data() {}   // Update prior to display

  virtual void find_ymin_ymax (double& ymin, double& ymax);
  
  base_data(int num_points);           // To allocate internal storage
  base_data(T* data, int num_points);  // To use external storage
  virtual ~base_data();
};


template<typename T> void base_data<T>::set_scale(double x_data_start, double x_data_step, double y_scale)
{
}


template<typename T> T base_data<T>::get_data(int index)
{
#ifdef DEBUG_BASE_DATA
  if(index<0)
    {
      printf("In base_data::get_data():  index less than zero.\n");
      abort();
    }
  else if(index>=num_points)
    {
      printf("In base_data::get_data():  index less than num_points.\n");
      abort();      
    }
#endif

  if(group_length==1)
    {
      return data[index];
    }
  else
    {
      int group_index = index / valid_per_group;
      int base_index  = group_index * group_length;
      int base_offset = index % valid_per_group;
      //printf("%d %d %d %d\n", group_index, base_index, base_offset, base_index+base_offset);
      return data[base_index + base_offset];
    }
}


template<typename T> base_data<T>::base_data(int num_points)
{
  this->data           =  new T[num_points];
  this->num_points     =  num_points;
  this->delete_data    =  true;
  this->direct_access  =  true;  // Often overridden in derived class

  group_length    = 1;
  valid_per_group = 1;

  this->g = 0;
  
  for(int i=0; i<num_points; i++)
    this->data[i] = 0;
}


template<typename T> base_data<T>::base_data(T* data, int num_points)
{
  this->data           =  data;
  this->num_points     =  num_points;
  this->delete_data    =  false;
  this->direct_access  =  true;  // Often overridden in derived class

  group_length    = 1;
  valid_per_group = 1;

  this->g = 0;
}


template<typename T> base_data<T>::~base_data()
{
  if(delete_data)
    delete [] data;
}


template<typename T> void base_data<T>::find_ymin_ymax(double& ymin, double& ymax)
{
  bool first = true;
  ymin = -1.0;
  ymax =  1.0;
  
  for(int pnt=0; pnt<num_points; pnt++)
    {
      double y = get_data(pnt);

      if(isfinite(y))
	{
	  if(first)
	    {
	      ymin = ymax = y;
	      first = false;
	    }
	  else
	    {
	      ymin = (y < ymin) ? y : ymin;
	      ymax = (y > ymax) ? y : ymax;
	    }
	}
    }  
}


#endif
