
#ifndef INPUT_EVENT_HANDLER_HH
#define INPUT_EVENT_HANDLER_HH

#include <stdlib.h>
#include <linux/input.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/types.h>
#include <dirent.h>
#include <stdio.h>
#include <stdlib.h>
#include <poll.h>
#include "my_event.hh"
#include "runnable.h"
#include "synchronized_data.h"

struct my_event_queue
{
  const static int MAX_EVENTS = 32;
  int event_in;
  int event_out;
  my_event event_rotating_buffer[MAX_EVENTS];
  my_event no_event;
  
  my_event_queue()  { event_in = 0; event_out = 0; no_event.type=EVENT_NONE; }

  my_event get_event()
  {
    if(event_out==event_in)
      return no_event;

    my_event m = event_rotating_buffer[event_out];
    event_out = (event_out + 1) % MAX_EVENTS;
    return m;
  }

  bool put_event(my_event& m)
  {
    int next_event_in = (event_in+1)%MAX_EVENTS;
    if(next_event_in==event_out)
      return false;  // No room
    event_rotating_buffer[event_in] = m;
    event_in = next_event_in;
    return true;
  }
};


class input_event_handler : runnable
{
  int maxx;
  int maxy;
  int current_listener;
  
  const static int MAX_LISTENERS = 20;

  int num_listeners;
  class input_event_listener* listeners[MAX_LISTENERS];

  void scan_for_input_devices();
  
 public:

  synchronized_data<my_event_queue> out;
  class display* disp;

  input_event_handler(int maxx, int maxy, display* disp);
  ~input_event_handler();

  void set_verbose();

  virtual void run();

  my_event wait_for_event();  // Will sleep until some event happens
  my_event check_for_event(); // Returns immediately; event may be EVENT_NONE
};



enum motion_state
  {
    MOTION_STATE_NONE,     // Not handling motion, waiting a touch
    MOTION_STATE_INIT,     // Got a touch, waiting 100ms to make sure another finger doesn't come down
    MOTION_STATE_HOLDING,  // Have decided how many fingers, but no significant motion yet.  Send hold time events, flush minor motion.
    MOTION_STATE_MOVING,   // Got a stable touch, moved away from it, processing move events
    MOTION_STATE_ERROR,    // A finger was released on a multitouch event; sent a release to client, awaiting one from device, flushing all motion
  };

enum key_state
  {
    KEY_STATE_NONE,    // Awaiting a press
    KEY_STATE_PRESSED  // Awaiting a release, sending hold time events, flushing all other key presses
  };

//
// One listener per input device.  The input_event_handler brings them all together
// into a single event stream for the application.
//
class input_event_listener : runnable
{
  input_event_handler*  ieh;
  
  const static int      wait_for_fingers_us  = 100000;
  const static int      motion_to_break_hold = 10;

  int                   fd;

  int                   touchpad_to_screen_x_offset;
  int                   touchpad_to_screen_x_numerator;
  int                   touchpad_to_screen_x_denominator;

  int                   touchpad_to_screen_y_offset;
  int                   touchpad_to_screen_y_numerator;
  int                   touchpad_to_screen_y_denominator;

  enum motion_state     old_motion_touch_state;
  enum motion_state     motion_touch_state;
  bool                  motion_touch_event;          // got a touch
  bool                  motion_release_event;        // got a release
  bool                  motion_event;                // got a motion event
  bool                  motion_finger_loss_event;    // lost a finger
  bool                  motion_finger_add_event;     // added a finger
  long long             motion_event_start_us;
  my_event              motion_starting_position;
  
  bool                  touchpad;
  bool                  mouse;
  my_event              me;
  my_event              me_at_touch;

  enum key_state        key_press_state;
  bool                  key_press_event;       // got a key press
  bool                  key_release_event;     // got a key release
  long long             key_event_start_us;

  bool                  wheel_event;
  
  int                   current_slot;
  
  const static int      MAX_SLOTS = 32;
  int                   touch_for_each_slot[MAX_SLOTS];
  int                   tracking_id_for_each_slot[MAX_SLOTS];

  int                   num_touches;
  
  static const char *ev_type[EV_CNT];
  static const char *ev_code_syn[SYN_CNT];
  static const char *ev_code_abs[ABS_CNT];
  static const char *ev_code_rel[REL_CNT];
  static const char *ev_code_key[KEY_CNT];
  static const char *ev_code_msc[MSC_CNT];

  void init();
  
public:

  char* filename;

  bool verbose;

  void print_event(struct input_event& ie);
  void send_event(my_event& me);
  
  input_event_listener(const char* devname, int screen_width, int screen_height, input_event_handler* ieh);
  ~input_event_listener();

  virtual void run();
  
  my_event& get_event();
};



#endif
