
#ifndef INPUT_EVENT_HANDLER_THREADED_HH
#define INPUT_EVENT_HANDLER_THREADED_HH

#include "runnable.h"
#include "synchronized_data.h"
#include <stdlib.h>
#include <linux/input.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/types.h>
#include <dirent.h>
#include <stdio.h>
#include <stdlib.h>

enum my_event_type
  {
   EVENT_NONE,
   EVENT_MOUSE_MOVE_SOURCE_ABS,
   EVENT_MOUSE_MOVE_SOURCE_REL, // Internally, relative coordinates.  Externally, absolute coordinates
   EVENT_MOUSE_BUTTON_PRESS,
   EVENT_MOUSE_BUTTON_RELEASE,
   EVENT_KEY_PRESS
  };


struct my_event
{
  my_event_type type;
  int           x;
  int           y;
  int           key_code;
  char          key_ascii;
};


struct my_event_queue
{
  const static int MAX_EVENTS = 32;
  int event_in;
  int event_out;
  my_event event_rotating_buffer[MAX_EVENTS];
  my_event no_event;
  
  my_event_queue()  { event_in = 0; event_out = 0; no_event.type=EVENT_NONE; }

  my_event get_event()
  {
    if(event_out==event_in)
      return no_event;

    my_event m = event_rotating_buffer[event_out];
    event_out = (event_out + 1) % MAX_EVENTS;
    return m;
  }

  bool put_event(my_event& m)
  {
    int next_event_in = (event_in+1)%MAX_EVENTS;
    if(next_event_in==event_out)
      return false;  // No room
    event_rotating_buffer[event_in] = m;
    event_in = next_event_in;
    return true;
  }
};


class input_event_handler_threaded
{
  int maxx;
  int maxy;
  int current_x;
  int current_y;
  
  const static int MAX_LISTENERS = 20;

  int num_listeners;
  class event_listener* listeners[MAX_LISTENERS];

  synchronized_data<my_event_queue> out;

  void fix_coordinates(my_event& me);

 public:
  input_event_handler_threaded(int maxx, int maxy);
  ~input_event_handler_threaded();

  my_event wait_for_event();  // Will sleep until some event happens
  my_event check_for_event(); // Returns immediately; event may be EVENT_NONE
};



class event_listener : runnable
{
  int fd;
  bool button_down;
  bool got_rel_movement;
  bool got_abs_movement;
  
  synchronized_data<my_event_queue>* out;

  static const char *ev_type[EV_CNT];
  static const char *ev_code_syn[SYN_CNT];
  static const char *ev_code_abs[ABS_CNT];
  static const char *ev_code_rel[REL_CNT];
  static const char *ev_code_key[KEY_CNT];
  static const char *ev_code_msc[MSC_CNT];

public:
  
  void print_event(struct input_event& ie);
  
  event_listener(const char* devname, synchronized_data<my_event_queue>* out);
  ~event_listener();

  void send_event(my_event& me);
  
  virtual void run();
};


char keycode_to_ascii(int code);

const char *event_listener::ev_type[EV_CNT]       = { 0 };
const char *event_listener::ev_code_syn[SYN_CNT]  = { 0 };
const char *event_listener::ev_code_abs[ABS_CNT]  = { 0 };
const char *event_listener::ev_code_rel[REL_CNT]  = { 0 };
const char *event_listener::ev_code_key[KEY_CNT]  = { 0 };
const char *event_listener::ev_code_msc[MSC_CNT]  = { 0 };



int range_limit(int x, int min, int max)
{
  if(x<min)
    x=min;
  if(x>=max)
    x=max-1;

  return x;
}



inline void event_listener::print_event(struct input_event& ie)
{
  char type[100];
  char code[100];

  if(ev_type[ie.type])
    sprintf(type, "type:%s", ev_type[ie.type]);
  else
    sprintf(type, "unknown type:%d", ie.type);

  if(ie.type==EV_ABS && ev_code_abs[ie.code])
    sprintf(code, "code:%s", ev_code_abs[ie.code]);
  else if(ie.type==EV_SYN && ev_code_syn[ie.code])
    sprintf(code, "code:%s", ev_code_syn[ie.code]);
  else if(ie.type==EV_REL && ev_code_rel[ie.code])
    sprintf(code, "code:%s", ev_code_rel[ie.code]);
  else if(ie.type==EV_KEY && ev_code_key[ie.code])
    sprintf(code, "code:%s", ev_code_key[ie.code]);
  else if(ie.type==EV_MSC && ev_code_msc[ie.code])
    sprintf(code, "code:%s", ev_code_msc[ie.code]);
  else
    sprintf(code, "unknown code:%d", ie.code);

  fprintf(stderr, "time:%ld.%06ld\t%s\t%s\tvalue:%d\n",
	  ie.time.tv_sec, ie.time.tv_usec, type,
	  code, ie.value);
}



inline event_listener::event_listener(const char* devname, synchronized_data<my_event_queue>* out)
{
  this->out = out;
  button_down = false;
  got_abs_movement = false;
  got_rel_movement = false;
  
  fd = open(devname, O_RDONLY);
  if(fd == -1)
    {
      printf("Error opening device \"%s\".", devname);
      exit(20);
    }


  ev_type[EV_SYN]       = "EV_SYN";
  ev_type[EV_KEY]       = "EV_KEY";
  ev_type[EV_REL]       = "EV_REL";
  ev_type[EV_ABS]       = "EV_ABS";
  ev_type[EV_MSC]       = "EV_MSC";
  ev_type[EV_SW]        = "EV_SW";
  ev_type[EV_LED]       = "EV_LED";
  ev_type[EV_SND]       = "EV_SND";
  ev_type[EV_REP]       = "EV_REP";
  ev_type[EV_FF]        = "EV_FF";
  ev_type[EV_PWR]       = "EV_PWR";
  ev_type[EV_FF_STATUS] = "EV_FF_STATUS";
  ev_type[EV_MAX]       = "EV_MAX";
  

  ev_code_syn[SYN_REPORT]    = "SYN_REPORT";
  ev_code_syn[SYN_CONFIG]    = "SYN_CONFIG";
  ev_code_syn[SYN_MT_REPORT] = "SYN_MT_REPORT";
  ev_code_syn[SYN_DROPPED]   = "SYN_DROPPED";
  ev_code_syn[SYN_MAX]       = "SYN_MAX";
  
  ev_code_abs[ABS_X]    = "ABS_X";
  ev_code_abs[ABS_Y]    = "ABS_Y";
  ev_code_abs[ABS_Z]    = "ABS_Z";
  ev_code_abs[ABS_RX]    = "ABS_RX";
  ev_code_abs[ABS_RY]    = "ABS_RY";
  ev_code_abs[ABS_RZ]    = "ABS_RZ";

  ev_code_rel[REL_X]      = "REL_X";
  ev_code_rel[REL_Y]      = "REL_Y";
  ev_code_rel[REL_Z]      = "REL_Z";
  ev_code_rel[REL_RX]     = "REL_RX";
  ev_code_rel[REL_RY]     = "REL_RY";
  ev_code_rel[REL_RZ]     = "REL_RZ";
  ev_code_rel[REL_HWHEEL] = "REL_WHEEL";
  ev_code_rel[REL_DIAL]   = "REL_DIAL";
  ev_code_rel[REL_WHEEL]  = "REL_WHEEL";
  ev_code_rel[REL_MISC]   = "REL_MISC";
  ev_code_rel[REL_MAX]    = "REL_MAX";
  
  ev_code_key[BTN_LEFT]    = "BTN_LEFT";
  ev_code_key[BTN_RIGHT]   = "BTN_RIGHT";
  ev_code_key[BTN_MIDDLE]  = "BTN_MIDDLE";
  ev_code_key[BTN_SIDE]    = "BTN_SIDE";
  ev_code_key[BTN_EXTRA]   = "BTN_EXTRA";
  ev_code_key[BTN_FORWARD] = "BTN_FORWARD";
  ev_code_key[BTN_BACK]    = "BTN_BACK";
  ev_code_key[BTN_TASK]    = "BTN_TASK";
  ev_code_key[KEY_MAX]     = "KEY_MAX";

  ev_code_msc[MSC_SERIAL]    = "MSC_SERIAL";
  ev_code_msc[MSC_PULSELED]  = "MSC_PULSELED";
  ev_code_msc[MSC_GESTURE]   = "MSC_GESTURE";
  ev_code_msc[MSC_RAW]       = "MSC_RAW";
  ev_code_msc[MSC_SCAN]      = "MSC_SCAN";
  ev_code_msc[MSC_TIMESTAMP] = "MSC_TIMESTAMP";
  
  start();
}


inline event_listener::~event_listener()
{
  close(fd);
  // thread is automatically terminated by base class destructor
}



inline void event_listener::run()
{
  int          dummy;
  input_event  ie;
  my_event     me;
  me.x = me.y = me.key_code = me.key_ascii = 0;
  
  for(;;)
    {
      dummy = read(fd, &ie, sizeof(ie));
      dummy = dummy;
      
      print_event(ie);
      
      if (ie.type == EV_SYN && ie.code == SYN_REPORT)
	{
	  // Linux events handle x and y separately, then do a syn_report.  So send a mouse move when there is a syn_report.

	  if(got_abs_movement)
	    {
	      me.type = EVENT_MOUSE_MOVE_SOURCE_ABS;
	      send_event(me);
	    }
	  if(got_rel_movement)
	    {
	      me.type = EVENT_MOUSE_MOVE_SOURCE_REL;
	      send_event(me);
	    }

	  // Touchscreen devices don't always indicate a press at the start of the move.  Fix this.
	  if(got_abs_movement && !button_down)
	    {
	      me.type = EVENT_MOUSE_BUTTON_PRESS;
	      send_event(me);
	      button_down = true;
	    }

	  got_abs_movement = false;
	  got_rel_movement = false;
	}
      else if(ie.type==EV_KEY)
	{
	  me.key_code = ie.code;
	  me.key_ascii = keycode_to_ascii(me.key_code);
	  
	  if(ie.code == BTN_LEFT && ie.value == 1)
	    {
	      me.type = EVENT_MOUSE_BUTTON_PRESS;
	      send_event(me);
	      button_down = true;
	    }
	  else if(ie.code == BTN_LEFT && ie.value == 0)
	    {
	      me.type = EVENT_MOUSE_BUTTON_RELEASE;
	      send_event(me);
	      button_down = false;
	    }
	  else if (ie.value == 1)
	    {
	      me.type = EVENT_KEY_PRESS;
	      send_event(me);
	    }
	}
      else if(ie.type==EV_REL)
	{
	  got_rel_movement = true;
	  
	  if (ie.code == REL_X)
	    me.x = ie.value;
	  
	  if (ie.code == REL_Y)
	    me.y = ie.value;
	}
      else if(ie.type==EV_ABS)
	{
	  got_abs_movement = true;
	  
	  if (ie.code == ABS_X)
	    me.x = ie.value;
	  else if (ie.code == ABS_Y)
	    me.y = ie.value;
	}
      else if(ie.type==EV_MSC)
	{
	}
      else
	{
	}
      
    }
  
}


void event_listener::send_event(my_event& me)
{
  out->lock();
  while(!out->data.put_event(me))
    out->wait_for_change();
  out->unlock();
}


bool strings_match_to_length_of_first(const char* s1, const char* s2)
{
  while(*s1 && *s1==*s2)
    {
      s1++;
      s2++;
    }

  if(!*s1)
    return true;

  return false;
}


input_event_handler_threaded::input_event_handler_threaded(int maxx, int maxy)
{
  // Create listeners for all of /dev/input/event*
  
  // struct dirent {
  //   ino_t          d_ino;       /* Inode number */
  //   off_t          d_off;       /* Not an offset; see below */
  //   unsigned short d_reclen;    /* Length of this record */
  //   unsigned char  d_type;      /* Type of file; not supported
  //                                  by all filesystem types */
  //   char           d_name[256]; /* Null-terminated filename */
  // };

  this->maxx = maxx;
  this->maxy = maxy;
  this->current_x = maxx/2;
  this->current_y = maxy/2;
  
  num_listeners = 0;

  for(;;)
    {
      DIR* dp = opendir("/dev/input");

      if(!dp)
	{
	  printf("Can't search directory \"/dev/input\".\n");
	  usleep(100000);
	  continue;
	}
      
      printf("Listening to event files:\n\n");
      
      for(;;)
	{
	  dirent *de = readdir(dp);
	  
	  if(!de)
	    break;
	  
	  if(strings_match_to_length_of_first("event", de->d_name))
	    {
	      char ename[200];
	      printf("  %s\n", de->d_name);
	      sprintf(ename, "/dev/input/%.100s", de->d_name);
	      
	      listeners[num_listeners++] = new event_listener(ename, &out);
	    }
	}
    }

  printf("\nEnd of event files.\n");
}


input_event_handler_threaded::~input_event_handler_threaded()
{
  for(int i=0; i<num_listeners; i++)
    {
      delete listeners[i];
    }
}


// Adjusts absolute coordinates to match screen size; fixes
// relative coordinates so they're absolute; adds current
// coordinates to all events.
void input_event_handler_threaded::fix_coordinates(my_event& me)
{
  if(me.type == EVENT_MOUSE_MOVE_SOURCE_ABS)
    {
      current_x = maxx * me.x / 32768;
      current_y = maxy * me.y / 32768;
    }
  else if(me.type == EVENT_MOUSE_MOVE_SOURCE_REL)
    {
      current_x += me.x;
      current_y += me.y;

      current_x = range_limit(current_x, 0, maxx);
      current_y = range_limit(current_y, 0, maxy);
    }

  me.x = current_x;
  me.y = current_y;
}

// Will sleep until some event happens
my_event input_event_handler_threaded::wait_for_event()
{
  my_event me;
  out.lock();
  for(;;)
    {
      me = out.data.get_event();
      if(me.type!=EVENT_NONE)
	break;
      out.wait_for_change();
    }
  out.unlock();

  fix_coordinates(me);
  
  return me;
}


// Returns immediately; event may be EVENT_NONE
my_event input_event_handler_threaded::check_for_event()
{  
  out.lock();
  my_event me = out.data.get_event();
  out.unlock();

  fix_coordinates(me);

  return me;
}



char keycode_to_ascii(int code)
{
  bool do_init = true;
  const static int MAXC = 100;
  static char code_to_key[MAXC];

  if(do_init)
    {
      for(int i=0; i<MAXC; i++)
	code_to_key[i] = 0;
      
      code_to_key[KEY_0] = '0';
      code_to_key[KEY_1] = '1';
      code_to_key[KEY_2] = '2';
      code_to_key[KEY_3] = '3';
      code_to_key[KEY_4] = '4';
      code_to_key[KEY_5] = '5';
      code_to_key[KEY_6] = '6';
      code_to_key[KEY_7] = '7';
      code_to_key[KEY_8] = '8';
      code_to_key[KEY_9] = '9';
      code_to_key[KEY_A] = 'A';
      code_to_key[KEY_B] = 'B';
      code_to_key[KEY_C] = 'C';
      code_to_key[KEY_D] = 'D';
      code_to_key[KEY_E] = 'E';
      code_to_key[KEY_F] = 'F';
      code_to_key[KEY_G] = 'G';
      code_to_key[KEY_H] = 'H';
      code_to_key[KEY_I] = 'I';
      code_to_key[KEY_J] = 'J';
      code_to_key[KEY_K] = 'K';
      code_to_key[KEY_L] = 'L';
      code_to_key[KEY_M] = 'M';
      code_to_key[KEY_N] = 'N';
      code_to_key[KEY_O] = 'O';
      code_to_key[KEY_P] = 'P';
      code_to_key[KEY_Q] = 'Q';
      code_to_key[KEY_R] = 'R';
      code_to_key[KEY_S] = 'S';
      code_to_key[KEY_T] = 'T';
      code_to_key[KEY_U] = 'U';
      code_to_key[KEY_V] = 'V';
      code_to_key[KEY_W] = 'W';
      code_to_key[KEY_X] = 'X';
      code_to_key[KEY_Y] = 'Y';
      code_to_key[KEY_Z] = 'Z';
      code_to_key[KEY_MINUS]      = '-';
      code_to_key[KEY_EQUAL]      = '=';
      code_to_key[KEY_TAB]        = '\t';
      code_to_key[KEY_LEFTBRACE]  = '[';
      code_to_key[KEY_RIGHTBRACE] = ']';
      code_to_key[KEY_ENTER]      = '\n';
      code_to_key[KEY_SEMICOLON]  = ';';
      code_to_key[KEY_APOSTROPHE] = '\'';
      code_to_key[KEY_GRAVE]      = '`';
      code_to_key[KEY_BACKSLASH]  = '\\';
      code_to_key[KEY_COMMA]      = ',';
      code_to_key[KEY_DOT]        = '.';
      code_to_key[KEY_SLASH]      = '/';
      code_to_key[KEY_KPASTERISK] = '*';
      code_to_key[KEY_SPACE]      = ' ';
      code_to_key[KEY_KP0]        = '0';
      code_to_key[KEY_KP1]        = '1';
      code_to_key[KEY_KP2]        = '2';
      code_to_key[KEY_KP3]        = '3';
      code_to_key[KEY_KP4]        = '4';
      code_to_key[KEY_KP5]        = '5';
      code_to_key[KEY_KP6]        = '6';
      code_to_key[KEY_KP7]        = '7';
      code_to_key[KEY_KP8]        = '8';
      code_to_key[KEY_KP9]        = '9';
      code_to_key[KEY_KPDOT]      = '.';
      code_to_key[KEY_KPPLUS]     = '+';
      code_to_key[KEY_KPMINUS]    = '-';
      code_to_key[KEY_KPSLASH]    = '/';
    }
  
  if(code<0||code>=MAXC)
    return 0;
  
  return code_to_key[code];
}


#endif
