
/*
 * Copyright (C) 2025 Bit by Bit Signal Processing LLC (https://bxbsp.com)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., http://www.fsf.org/about/contact/
 *
 */


#include "parse_support.hh"
#include <ctype.h>
#include <unistd.h>

int isInString(char c, const char* s)
{
  if(!s)
    {
      printf("Null String Error in isInString in parse_support.cc\n");
      exit(10);
    }

  while(*s)
    {
      if(*s++==c)
	return 1;
    }

  return 0;
}

int isNotInString(char c, const char* s)
{
  return !isInString(c,s);
}

bool equalToLengthOfFirst(const char* s1, const char* s2)
{
  if(!s1 || !s2)
    {
      printf("ERROR:  got null pointers in equalToLengthOfFirst()!\n");
      exit(20);
    }

  while(*s1)
    if(*s1++ != *s2++)
      return false;

  return true;
}


int skipChars(readstream& a, const char* chars)
{
  int c;

  do
    {
      c = a.getchar();
    }
  while(c!=EOF && isInString(c, chars));

  a.ungetchar();
  return c;
}

int skipUntilChars(readstream& a, const char* chars)
{
  int c;

  do
    {
      c = a.getchar();
    }
  while(c!=EOF && !isInString(c, chars));

  a.ungetchar();
  return c;
}

int readInt(readstream& a)
{
  char buff[100];
  int c;
  int negative = 0;

  buff[0]=0;

  skipChars(a, " \t");

  if(a.getchar()=='-')
    negative=1;
  else
    a.ungetchar();

  c = readStringKnownChars(a, buff, "0123456789", 100);

  if(c==100)
    {
      printf("Size exceeded in readInt in parse_support.cc\n");
      parse_error(a);
    }

  if(buff[0]==0)
    {
      printf("Invalid integer!\n");
      parse_error(a);
    }
  
  if(negative)
    return -atoi(buff);
  else
    return atoi(buff);
}

long readLong(readstream& a)
{
  char buff[100];
  int c;
  int negative = 0;

  skipChars(a, " \t");

  if(a.getchar()=='-')
    negative=1;
  else
    a.ungetchar();

  c = readStringKnownChars(a, buff, "0123456789", 100);

  if(c==100)
    {
      printf("Size exceeded in readLong in parse_support.cc\n");
      parse_error(a);
    }

  if(buff[0]==0)
    {
      printf("Invalid integer!\n");
      parse_error(a);
    }
  
  if(negative)
    return -atol(buff);
  else
    return atol(buff);
}

float readFloat(readstream& a)
{
  char buff[100];
  int num = 0;

  buff[0] = 0;

  skipChars(a, " \t");

  num += readStringKnownChars(a, &buff[num], "-", 100-num);
  num += readStringKnownChars(a, &buff[num], "0123456789", 100-num);
  num += readStringKnownChars(a, &buff[num], ".", 100-num);
  num += readStringKnownChars(a, &buff[num], "0123456789", 100-num);
  num += readStringKnownChars(a, &buff[num], "eE", 100-num);
  num += readStringKnownChars(a, &buff[num], "0123456789", 100-num);

  if(num==100)
    {
      printf("Size exceeded in readFloat in parse_support.cc\n");
      parse_error(a);
    }

  if((buff[0]==0)                                   || 
     (buff[0]=='-' && buff[1]==0)                   ||
     (buff[0]=='-' && buff[1]=='e')                 ||
     (buff[0]=='-' && buff[1]=='E')                 ||
     (buff[0]=='.' && buff[1]==0)                   ||
     (buff[0]=='.' && buff[1]=='e')                 ||
     (buff[0]=='.' && buff[1]=='E')                 ||
     (buff[0]=='e')                                 ||
     (buff[0]=='E')                                 ||
     (buff[0]=='-' && buff[1]=='.' && buff[2]=='e') ||
     (buff[0]=='-' && buff[1]=='.' && buff[2]=='E') ||
     (buff[0]=='-' && buff[1]=='.' && buff[2]==0))
    {
      printf("Invalid floating point number \"%s\"!\n", buff);
      parse_error(a);
    }
  
  return atof(buff);
}

int readString(readstream& a, char* buff, const char* terms, int maxsize)
{
  int i;
  int c;

  for(i=0; i<maxsize; i++)
    {
      c = a.getchar();
      if(c==EOF || isInString(c, terms))
	break;

      buff[i]=c;
    }

  a.ungetchar();
  buff[i]=0;
  return i;
}

int readStringWithEscape(readstream& a, char* buff, const char* terms, const char escape, int maxsize)
{
  int i;
  int c;

  for(i=0; i<maxsize; i++)
    {
      c = a.getchar();
      if(c==escape)
	{
	  c = a.getchar();
	}
      else
	{	  
	  if(isInString(c, terms))
	    break;
	}

      if(c==EOF)
	break;
	
      buff[i]=c;
    }

  a.ungetchar();
  buff[i]=0;
  return i;
}

bool validVariable(char* s)
{
  if(!s)
    return false;

  if( !isalpha(*s) && (*s != '_') )
    return false;
  
  s++;

  while(*s)
    {
      if( !isalnum(*s) && (*s != '_') )
	{
	  return false;
	}

      s++;
    }

  return true;
}

int readVariable(readstream& a, char* buff, const char* terms, int maxsize)
{
  int num = readString(a, buff, terms, maxsize);
  char* s = buff;

  if(num==0)
    return 0;

  if( !isalpha(*s) && (*s != '_') )
    {
      printf("Invalid first character in variable \"%s\"!\n", buff);
      parse_error(a);
    }

  s++;

  while(*s)
    {
      if( !isalnum(*s) && (*s != '_') )
	{
	  printf("Invalid character \"%c\" in variable \"%s\"!\n", *s, buff);
	  parse_error(a);
	}

      s++;
    }

  return num;
}

int readMultipinVariable(readstream& a, char* buff, const char* terms, int maxsize)
{
  int num = readString(a, buff, terms, maxsize);
  char* s = buff;

  if(num==0)
    return 0;

  if(!isalpha(*s++))
    {
      printf("Invalid variable \"%s\"!\n", buff);
      parse_error(a);
    }

  while(*s)
    {
      char x = *s++;
      if(x!='_' && !isalnum(x))
	{
	  if(x=='[')
	    {
	      while(isdigit(*s))
		s++;
	      if(*s!=':' && *s!=']')
		{
		  printf("C Invalid variable \"%s\"!\n", buff);
		  parse_error(a);
		}
	      if(*s!=']')
		{
		  s++;
		  while(isdigit(*s))
		    s++;
		}
	      if(*s!=']' || s[1]!=0)
		{
		  printf("B Invalid variable \"%s\"!\n", buff);
		  printf("s[0]=%c(%d), s[1]=%c(%d)\n", *s, (int)*s, s[1], (int)s[1]);
		  parse_error(a);
		}
	      break;
	    }

	  printf("A Invalid variable \"%s\"!\n", buff);
	  parse_error(a);
	}
    }

  return num;
}



//
// Allow special exception, where if termination is on a space/tab,
// will skip over spaces and tabs to look for a '[' character.  If
// such a character ends the spaces or tabs, the spaces/tabs will be
// skipped and not treated as terms.
//
int readMultipinVariableTD(readstream& a, char* buff, const char* terms, int maxsize)
{
  int c;
  int num = readString(a, buff, terms, maxsize);
  char* s = buff;

  c=skipChars(a, " \t");
  if(c=='[')
    {
      num += readString(a, buff+num, terms, maxsize-num);
    }

  if(num==0)
    return 0;

  if(!isalpha(*s++))
    {
      printf("Invalid variable \"%s\"!\n", buff);
      parse_error(a);
    }

  while(*s)
    {
      char x = *s++;
      if(x!='_' && !isalnum(x))
	{
	  if(x=='[')
	    {
	      while(isdigit(*s))
		s++;
	      if(*s!=':' && *s!=']')
		{
		  printf("C Invalid variable \"%s\"!\n", buff);
		  parse_error(a);
		}
	      if(*s!=']')
		{
		  s++;
		  while(isdigit(*s))
		    s++;
		}
	      if(*s!=']' || s[1]!=0)
		{
		  printf("B Invalid variable \"%s\"!\n", buff);
		  printf("s[0]=%c(%d), s[1]=%c(%d)\n", *s, (int)*s, s[1], (int)s[1]);
		  parse_error(a);
		}
	      break;
	    }

	  printf("A Invalid variable \"%s\"!\n", buff);
	  parse_error(a);
	}
    }

  return num;
}


int readStringKnownChars(readstream& a, char* buff, const char* chars, int maxsize)
{
  int i;
  int c;

  for(i=0; i<maxsize-1; i++)
    {
      c = a.getchar();
      if(c==EOF || isNotInString(c, chars))
	break;

      buff[i]=c;
    }

  a.ungetchar();
  buff[i]=0;
  return i;
}

//
// Checks if the upcoming characters match the given string.
// If so, file is positioned after the characters and true
// is returned.  If not, file position is not moved and
// false is returned.
//
int matchString(readstream& a, const char* s)
{
  int i=0;
  int c;

  if(!s)
    {
      printf("Null pointer in matchString in parse_support!\n");
      parse_error(a);
    }

  while(*s)
    {
      c=a.getchar();
      i++;

      //printf("Matching %c to %c (char %d)\n", *s, c, i);
      
      if(*s++!=c)
	{
	  a.ungetchars(i);
	  return 0;
	}
    }

  return 1;
}

//
// Like matchstring, but requires that there be no alpha characters immediately following
// the string.
//
int matchStringAlpha(readstream& a, const char* s)
{
  int match1 = matchString(a, s);
  if(match1)
    {
      int c = a.getchar();
      if(isalpha(c) || c=='_')
	{
	  a.ungetchars(strlen(s)+1);
	  return 0;
	}
      a.ungetchar();
    }

  return match1;
}

//
// Case insensitive version of matchString().
//
int matchStringI(readstream& a, const char* s)
{
  int i=0;
  int c;

  if(!s)
    {
      printf("Null pointer in matchString in parse_support!\n");
      parse_error(a);
    }

  while(*s)
    {
      c=a.getchar();
      i++;

      if(tolower(*s++)!=tolower(c))
	{
	  a.ungetchars(i);
	  return 0;
	}
    }

  return 1;
}

int skipCommentsCpp(readstream& a)
{
  int c, cc;

  for(;;)
    {
      cc = skipChars(a, " \t\n\r");
  
      if(matchString(a, "//"))
	{
	  skipUntilChars(a, "\n");
	}
      else if(matchString(a, "/*"))
	{
	  for(;;)
	    {
	      c = skipUntilChars(a, "*");

	      while(c=='*')
		c = a.getchar();

	      if(c=='/' || c==EOF)
		break;
	    }
	}
      else
	break;
    }

  return cc;
}


int skipCommentsUCF(readstream& a)
{
  int cc;

  for(;;)
    {
      cc = skipChars(a, " \t\n\r");
  
      if(cc=='#' || matchString(a, "//"))
	{
	  skipUntilChars(a, "\n");
	}
      else
	break;
    }

  return cc;
}


int skipCommentsTD(readstream& a)
{
  int c, cc;

  for(;;)
    {
      cc = skipChars(a, " \t\n\r");
  
      if(matchString(a, "//"))
	{
	  skipUntilChars(a, "\n");
	}
      else if(matchString(a, "#"))
	{
	  skipUntilChars(a, "\n");
	}
      else if(matchString(a, "/*"))
	{
	  for(;;)
	    {
	      c = skipUntilChars(a, "*");

	      while(c=='*')
		c = a.getchar();

	      if(c=='/' || c==EOF)
		break;
	    }
	}
      else
	break;
    }

  return cc;
}


int skipCommentsSh(readstream& a)
{
  int c;

  for(;;)
    {
      c = skipChars(a, " \t\n\r");
  
      if(c=='#')
	{
	  skipUntilChars(a, "\n");
	}
      else
	break;
    }

  return c;
}

void parse_error(readstream& r)
{
  //  int count=0;
  //  int c;
  char buff[1000];

  char* xx = getcwd(buff, 1000);

  printf("\n\n%s/%s:%d: error :\nParse error was on the line above, see further above for more detail.\n\n", 
	 xx,
	 r.getfilename(),
	 r.getlinenumber());

  printf("Text leading up to the error:\n\n");

  int numc = r.rewindToStartOfLine();

  for(int i=0; i<numc-1; i++)
    printf("%c", r.getchar());
  
  printf("\n\n");
	 
  
  printf("Following text is:\n\n");

  int count = 0;
  while(count<100)
    {
      count++;
      int c=r.getchar();
      if(c==EOF)
	{
	  printf("\nEOF\n");
	  break;
	}

      printf("%c", c);
      if(c=='\n')
	break;
    }
  

  printf("\n");

  exit(10);
}


void parse_warning(readstream& r, char* s)
{
  printf("WARNING: %s in file \"%s\" on line %d.\n", 
	 s,
	 r.getfilename(),
	 r.getlinenumber());
}


#define MAX_STRING 10000

String readQuotedString(readstream& a)
{
  char temp[MAX_STRING];
  int c;

  c = a.getchar();
  if(c!='"')
    {
      printf("Expecting quoted string.\n");
      parse_error(a);
    }

  c = readString(a, temp, "\"\n", MAX_STRING);
  if(c==MAX_STRING)
    {
      a.ungetchars(MAX_STRING);
      printf("Size exceeded in readQuotedString in parse_support.cc\n");
      parse_error(a);
    }

  c = a.getchar();
  if(c!='"')
    {
      printf("Problem reading terminating quote.\n");
      parse_error(a);
    }
  
  return String(temp);
}

String readExtendedQuotedString(readstream& a)
{
  char temp[2000];
  int c;

  c = a.getchar();
  if(c!='"')
    {
      printf("Expecting quoted string.\n");
      parse_error(a);
    }

  c = readStringWithEscape(a, temp, "\"", '\\', 2000);
  if(c==2000)
    {
      a.ungetchars(2000);
      printf("Size exceeded in readQuotedString in parse_support.cc\n");
      parse_error(a);
    }

  c = a.getchar();
  if(c!='"')
    {
      printf("Problem reading terminating quote.\n");
      parse_error(a);
    }
  
  return String(temp);
}

String readPossiblyQuotedString(readstream& r, const char* terms)
{
  char temp[200];
  int index=0;
  int quoteflag = 0;
  int c;

  for(;;)
    {
      c=r.getchar();
      if(c==EOF)
	{
	  r.ungetchar();
	  break;
	}

      if(isInString(c, terms) && !quoteflag)
	{
	  r.ungetchar();
	  break;
	}

      if(c=='"')
	{
	  quoteflag=!quoteflag;
	}
      else
	{
	  temp[index++] = c;

	  if(index==200)
	    {
	      r.ungetchars(200);
	      printf("String length exceeded in pmd_parse_quotedname!\n");
	      parse_error(r);
	    }
	}
    }
  temp[index]=0;

  return String(temp);
}


int skipCommentsVerilog(readstream& a)
{ 
  return skipCommentsCpp(a);
}

int skipCommentsVerilogNoNewline(readstream& a)
{
  int c, cc;

  for(;;)
    {
      cc = skipChars(a, " \t");
  
      if(matchString(a, "//"))
	{
	  cc = skipUntilChars(a, "\n");
	  return cc;
	}
      else if(matchString(a, "/*"))
	{
	  for(;;)
	    {
	      c = skipUntilChars(a, "*\n\r");

	      if(c=='\n' || c=='\r')
		{
		  printf("\nComments crossing multiple lines aren't permitted on this line.\n"
			 "This is true on some single-line statements, such as `define statements.\n");
		  parse_error(a);
		}

	      while(c=='*')
		c = a.getchar();

	      if(c=='/' || c==EOF)
		break;
	    }
	}
      else
	break;
    }

  return cc;
}
