
// SPDX-License-Identifier: CC-BY-NC-SA-4.0
//
// Copyright (C) 2025 Bit by Bit Signal Processing LLC  (https://bxbsp.com)
//
// This work is placed under the "Creative Commons Attribution
// NonCommercial ShareAlike 4.0 International" license, known
// by the shortened acronym "CC-BY-NC-SA-4.0".
//
// This work is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//
// A CC-BY-NC-SA-4.0 license allows you to use, distribute, and modify
// this work, so long as such uses are non-commercial in nature,
// so long as any derived works are offered on the same terms,
// and so long as attribution is given to the original author.
// For further details, see the Creative Commons License
// "CC-BY-NC-SA-4.0".
//
// You should have received a copy of the CC-BY-NC-SA-4.0 license
// along with this work. If not, see
// <https://creativecommons.org/licenses/by-nc-sa/4.0/>.
//


#ifndef HWALLOC_H
#define HWALLOC_H

#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/mman.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

//
// These routines are meant to map a stucture or array into physical memory.
//
// Example to create a structure "d" mapping to physical memory at 0xA0000000:
//
//  struct dev { uint32_t a, uint16_t b };
//
//  dev* d = hwalloc<dev>(0xA0000000);
//
//    ...
//
//  hwfree(d);
//
//
//
// Example to create an array of 100 32-bit ints at physical memory address 0xB0000000:
//
//
//  uint32_t* d = hwalloc<uint32_t>(0xA0000000, 100);
//
//    ...
//
//  hwfree(d, 100);
//

template <typename T> volatile T* hwalloc(uint32_t hwaddress, int count=1)
{
  if(count<=0)
    {
      printf("ERROR: In hwalloc, zero or negative count.\n");
      abort();
    }

  if((hwaddress&3)!=0)
    {
      printf("ERROR: In hwalloc, hwaddress 0x%08x is misaligned.\n", hwaddress);
      abort();
    }

  //printf("INFO: In hwalloc, hwaddress is 0x%08x, count is %d\n", hwaddress, count);
  //fflush(stdout);
  //usleep(1000);

  int fd = open("/dev/mem",O_RDWR|O_SYNC);
  if(fd<0)
    {
      printf("ERROR: In hwalloc, can't open /dev/mem\n");
      exit(20);
    }
  
  volatile T* object = (volatile T*) mmap(0, count * sizeof(T), PROT_READ|PROT_WRITE, MAP_SHARED, fd, hwaddress);
  if(!object)
    {
      printf("ERROR: In hwalloc, can't map memory to access physical addresses\n");
      exit(20);
    }

  close(fd);

  return object;
}



template <typename T> void hwfree(volatile T* object, int count=1)
{
  munmap((void*)object, count * sizeof(T));
}


#endif
