
// SPDX-License-Identifier: CC-BY-NC-SA-4.0
//
// Copyright (C) 2026 Bit by Bit Signal Processing LLC (https://bxbsp.com)
//
// This work is placed under the "Creative Commons Attribution
// NonCommercial ShareAlike 4.0 International" license, known
// by the shortened acronym "CC-BY-NC-SA-4.0".
//
// This work is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//
// A CC-BY-NC-SA-4.0 license allows you to use this work for
// noncommercial purposes so long as attribution is made to the
// original author.  Modified versions of this work may be distributed,
// but only under the same license.  For further details, see the
// Creative Commons License "CC-BY-NC-SA-4.0".
//
// You should have received a copy of the CC-BY-NC-SA-4.0 license
// along with this work. If not, see
// <https://creativecommons.org/licenses/by-nc-sa/4.0/>.
//


#include "displays.hh"
#include "window.hh"
#include "global_beep.hh"
#include "draw_text.hh"
#include "event_types.hh"
#include "send_events.hh"
#include "all_svg.hh"

#define GRAPH_PANE   0
#define MENU_PANE    1
#define SCRATCH_PANE 2
#define NUM_PANES    3

class intro : public window
{
  const static int blink_interval = 100;
  int              blink_count;
  bool             blink_on;
  float            ratio;

  const static int pointer_display_interval = 100;
  int              last_num;
  int              last_x[MAX_TOUCHES];
  int              last_y[MAX_TOUCHES];
  int              last_count;
  
public:

  //virtual void     calculate_width_and_height(int& calc_width, int& calc_height, int desired_height, float ratio);

  virtual bool     handle_event(my_event& me);

  virtual void     layout();
  virtual void     draw_dirty();
  virtual void     draw_dynamic();

                   intro();
                   ~intro();
};


bool intro::handle_event(my_event& me)
{
  if(me.type == EVENT_POINTER_MOVE || me.type == EVENT_MOVE)
    {
      last_num = me.num_touches;
      for(int i=0; i<last_num; i++)
	{
	  last_x[i] = me.c[i].x - offset.x;
	  last_y[i] = me.c[i].y - offset.y;
	}
      last_count = pointer_display_interval;
    }

  return true;  // Event handled
}


void intro::layout()
{
  float designed_height = 1080.0;
  float designed_width  = 1920.0;
  float ratio_x = width / designed_width;
  float ratio_y = height / designed_height;
  ratio = (ratio_x<ratio_y) ? ratio_x : ratio_y;

#ifdef NOTDEF
  int size    = 400 * ratio;
  int xmargin = 80 * ratio;
  int ymargin = 40 * ratio;
  draw_svg_from_data(BxB_svg, BxB_svg_length, width - size - xmargin, ymargin, size, size);
#endif

  set_text_size(90 * ratio);

  draw_text("Test App",
	    WHITE,
	    width/2-100*ratio,
	    220*ratio,
	    DRAW_TEXT_X_CENTER|DRAW_TEXT_Y_TOP|DRAW_TEXT_ROTATE_0);
  

      
  dt_colors textcolors = { GREEN7, REDRED8, WHITE };
  
  //
  // Saved as examples of multi colors
  //
  // "Signal Generator Demo\x0B (Forthcoming; now 10MHz sine)",
  // "\x0C" "Full version is for sale.\x0B (Forthcoming)\x0C  Also ask about:",
  
  set_text_size(50*ratio);
  
  draw_multicolored_text("\x0C" "White" "\x0B" " Red" "\x0A" " Green",
			 0,
			 textcolors,
			 width/2-100*ratio,
			 570*ratio,
			 DRAW_TEXT_X_CENTER|DRAW_TEXT_Y_TOP|DRAW_TEXT_ROTATE_0);
    
  layout_dirty = false;
  dirty = true;
}


void intro::draw_dirty()
{
  dirty = false;
}


void intro::draw_dynamic()
{
  if(blink_count>=blink_interval)
    {
      blink_on = !blink_on;
      blink_count = 0;
    }
  else
    {
      blink_count++;
    }


  if(blink_on)
    {      
      set_text_size(40 * ratio);

      draw_text("Blinking Text",
		GREEN,
		65 * ratio + 10,
		64 * ratio + 10,
		DRAW_TEXT_X_LEFT|DRAW_TEXT_Y_CENTER|DRAW_TEXT_ROTATE_0);
    }

  if(last_count)
    {
      last_count--;

      color colors[]   = { WHITE, RED, GREEN, BLUE, ORANGE, BLUEGREEN, PURPLE, YELLOW, GREY };
      int   num_colors = sizeof(colors) / sizeof(colors[0]);
      
      point p0, p1;

      for(int i=0; i<last_num; i++)
	{
	  color c = colors[i%num_colors];
      
	  p0.x = 0;
	  p1.x = width;
	  p0.y = last_y[i];
	  p1.y = last_y[i];
	  
	  draw_line_horizontal(p0, p1, c);
	  
	  p0.x = last_x[i];
	  p1.x = last_x[i];
	  p0.y = 0;
	  p1.y = height;
	  draw_line_vertical(p0, p1, c);
	}
    }
}


intro::intro() : window(BLACK)
{
  blink_count  = 0;
  blink_on     = false;
  ratio        = 1;

  last_count   = 0;
}


intro::~intro()
{
}


int main()
{
  displays* d = new displays(NUM_PANES);


  intro* intro_window = new intro();
  d->add(intro_window);


  d->draw(); // Draw static, dynamic, and flip the page
  

  for(long int count = 0;; count++)
    {
      //printf("count=%ld\n", count);
      
      d->get_and_process_events();  // Fetch, distribute, and handle mouse and key events

      //d->update_data();    // Update data that needs updates whether draws occur or not.

      d->draw(); // Draw static, dynamic, and flip the page

      
      usleep(400);
    }
  
  return 0;
}


