
// SPDX-License-Identifier: CC-BY-NC-SA-4.0
//
// Copyright (C) 2026 Bit by Bit Signal Processing LLC (https://bxbsp.com)
//
// This work is placed under the "Creative Commons Attribution
// NonCommercial ShareAlike 4.0 International" license, known
// by the shortened acronym "CC-BY-NC-SA-4.0".
//
// This work is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//
// A CC-BY-NC-SA-4.0 license allows you to use this work for
// noncommercial purposes so long as attribution is made to the
// original author.  Modified versions of this work may be distributed,
// but only under the same license.  For further details, see the
// Creative Commons License "CC-BY-NC-SA-4.0".
//
// You should have received a copy of the CC-BY-NC-SA-4.0 license
// along with this work. If not, see
// <https://creativecommons.org/licenses/by-nc-sa/4.0/>.
//

//

#ifndef MULTILINGUA_HH
#define MULTILINGUA_HH

#include "display.hh"

class multilingua
{
public:
  
  constexpr static int MAX_LANGUAGES = 32;

private:
	      
  char32_t* texts[MAX_LANGUAGES];
  static int current_language;

public:
  
  template <typename T> void set_text(const T* text, int language=0);
  template <typename T> void append_text(const T* text, int language=0);
  const char32_t* get_text(int language) const;
  const char32_t* get_text() const { return get_text(current_language); };
  
  operator const char32_t*() const;

  multilingua();
  template <typename T> multilingua(const T* text, int language=0);
  ~multilingua();

  static void set_language(display* d, int language_number);
  static int  get_current_language();  

  multilingua(const multilingua& ml);
  void operator=(const multilingua& ml);
};


#include "unalingua.hh"

template <typename T> multilingua::multilingua(const T* text, int language)
{
  for(int i=0; i<MAX_LANGUAGES; i++)
    texts[i] = 0;

  set_text(text, language);
}

template <typename T> void multilingua::set_text(const T* text, int language)
{
  int length = 0;
  for(;;length++)
    {
      if(!text[length])
	break;
    }

  if(texts[language])
    delete [] texts[language];

  texts[language] = new_UTF32(text);
}


template <typename T> void multilingua::append_text(const T* text, int language)
{
  if(!texts[language])
    {
      set_text(text, language);
      return;
    }
  
  int old_length = 0;
  for(;;old_length++)
    {
      if(!texts[language][old_length])
	break;
    }

  char32_t* added_text = new_UTF32(text);
  
  int length = 0;
  for(;;length++)
    {
      if(!added_text[length])
	break;
    }

  char32_t* new_text = new char32_t[old_length + length + 1];
  

  int i;
  for(i=0; i<old_length; i++)
    new_text[i] = texts[language][i];

  for(int j=0; j<length; j++)
    new_text[i+j] = added_text[j];

  new_text[length+old_length] = 0;

  delete [] texts[language];
  delete [] added_text;

  texts[language] = new_text;
}


#endif
