
// SPDX-License-Identifier: CC-BY-NC-SA-4.0
//
// Copyright (C) 2026 Bit by Bit Signal Processing LLC (https://bxbsp.com)
//
// This work is placed under the "Creative Commons Attribution
// NonCommercial ShareAlike 4.0 International" license, known
// by the shortened acronym "CC-BY-NC-SA-4.0".
//
// This work is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//
// A CC-BY-NC-SA-4.0 license allows you to use this work for
// noncommercial purposes so long as attribution is made to the
// original author.  Modified versions of this work may be distributed,
// but only under the same license.  For further details, see the
// Creative Commons License "CC-BY-NC-SA-4.0".
//
// You should have received a copy of the CC-BY-NC-SA-4.0 license
// along with this work. If not, see
// <https://creativecommons.org/licenses/by-nc-sa/4.0/>.
//
//

#ifndef MY_EVENT_HH
#define MY_EVENT_HH

enum my_event_type
  {
    //
    // These are non-events.
    //
    EVENT_NONE,
    EVENT_POINTER_MOVE,  // Mouse only, movement without a button pressed (only to move pointer on screen)

    //
    // These describe a normal sequence of an interaction.  First there is a touch (or mouse button press),
    // then it may hold in position (or not), then it moves, then it is released.  Almost all pointing events
    // are from this sequence.
    //
    // In the event stucture, there are different variables for mouse or touchpanel events from this normal
    // sequence:
    //
    //        source_mouse          -- true for a mouse, false for a touchpanel.
    //        num_touches           -- 1 for a mouse.  number of fingers for a touchpanel.
    //        c                     -- coordinates for the touches, valid for touch, hold, move, and release.
    //        mouse_buttons_pressed -- mask saying which mouse buttons are pressed.  0 for touchpanel
    //        hold_count            == length of hold in 100ms, after touch and before movement.
    //
    EVENT_TOUCH,         // Touch, or mouse button pressed
    EVENT_HOLD,          // Touch one or more positions and don't move.  How long is in hold_count
    EVENT_MOVE,          // Only sends move events when there's a touch or a mouse button pressed
    EVENT_RELEASE,

    //
    // Pointing events not in the normal sequence. EVENT_WHEEL uses the value field.  positive values are
    // scrolling forward.  Negative are scrolling backward.  Typical numbers are -1 or +1, but fast
    // scrolls can give higher values.
    //
    EVENT_WHEEL,

    //
    // Keyboard events.  hold_count is also used for a key hold.
    //
    EVENT_KEY_PRESS,
    EVENT_KEY_HOLD,
    EVENT_KEY_RELEASE,
  };


// If there are more than MAX_TOUCHES touches on a touchpanel, they won't be reported.
#define MAX_TOUCHES 5

struct screen_coordinates
{
  int x;
  int y;

  screen_coordinates() { x=0; y=0; }
};


// For mouse_buttons_pressed
const static int  MOUSE_BUTTON_LEFT   = 1;
const static int  MOUSE_BUTTON_MIDDLE = 2;
const static int  MOUSE_BUTTON_RIGHT  = 4;


//
// My input_event_handler code will wait a bit (100ms?)after a touch (for a touchscreen) to determine how many fingers
// have touched.  After that, the number of fingers touching won't change in reporting to the calling code.
//
// If additional fingers touch, they won't be reported, but events will continue with the touching fingers like the
// additional fingers never touched.  If one or more touch points release, it will be counted as the end of the event,
// and remaining events will be silently dropped until all fingers leave the screen.
//
// Mouse buttons pressed are handled in the same way.
//

struct my_event
{
  my_event_type       type;
  int                 num_touches;            // Will be 1 for a mouse, number of fingers touching the screen for a touchscreen.
  screen_coordinates  c [MAX_TOUCHES];
  int                 key_code;               // key codes are defined in /usr/include/linux/input-event-codes.h
  char                key_ascii;              // translated in my code (input_event_handler.h), so perhaps imperfectly.
  unsigned char       mouse_buttons_pressed;  // Which mouse buttons are pressed to generate TOUCH, HOLD, MOVE, and RELEASE events 
  int                 count;                  // hold count in 100ms increments.  Also used for scroll count.
  bool                source_mouse;           // Source is like a mouse. i.e. needs a screen pointer drawn, has buttons, moves without touch
  long long           time_in_us;             // Timestamp of the event in us

  my_event() { type = EVENT_NONE; num_touches = 0; for(int i=0; i<MAX_TOUCHES; i++) { c[i].x = 0; c[i].y = 0;} key_code = 0; key_ascii = 0; mouse_buttons_pressed = 0; count = 0; source_mouse = false; time_in_us =0; }
};


void print_event(my_event& me);

#endif
