
// SPDX-License-Identifier: CC-BY-NC-SA-4.0
//
// Copyright (C) 2026 Bit by Bit Signal Processing LLC (https://bxbsp.com)
//
// This work is placed under the "Creative Commons Attribution
// NonCommercial ShareAlike 4.0 International" license, known
// by the shortened acronym "CC-BY-NC-SA-4.0".
//
// This work is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//
// A CC-BY-NC-SA-4.0 license allows you to use this work for
// noncommercial purposes so long as attribution is made to the
// original author.  Modified versions of this work may be distributed,
// but only under the same license.  For further details, see the
// Creative Commons License "CC-BY-NC-SA-4.0".
//
// You should have received a copy of the CC-BY-NC-SA-4.0 license
// along with this work. If not, see
// <https://creativecommons.org/licenses/by-nc-sa/4.0/>.
//

#include "kiosk/window.hh"
#include "kiosk/multiwindow.hh"
#include "kiosk/button.hh"
#include "kiosk/unalingua.hh"

#include "button_demo.hh"
#include "languages.hh"

void button_push_function(button*b, void* v);


class button_demo_window : public multiwindow
{
  constexpr static int NUM_BUTTONS = 6;
  button* b[NUM_BUTTONS];

  virtual void layout();
  virtual void draw_dirty();
  virtual void draw_dynamic();

  void my_draw_box(color c, int x, int y, int w, int h);

public:

  const static int text_display_interval    = 100;

  unalingua        last_text;
  int              last_text_count;
  dt_colors        last_text_colors;

  button_demo_window();
  ~button_demo_window();
};

window* create_button_demo_window()
{
  return new button_demo_window();
}


button_demo_window::~button_demo_window()
{
  for(int i=0; i<NUM_BUTTONS; i++)
    if(b[i])
      delete b[i];
}

button_demo_window::button_demo_window()
{
  last_text_count = 0;

  for(int i=0; i<NUM_BUTTONS; i++)
    b[i] = 0;

  dt_colors c  = { WHITE };
  dt_colors cf = { RED   };
  
  dt_colors cb0  = { GREEN2, RED };
  dt_colors cb0f = { RED,   GREEN2 };

  dt_colors cb1  = { RED, ORANGE, YELLOW, GREEN, BLUE, PURPLE, WHITE };
  dt_colors cb1f = { PURPLE, BLUE, GREEN, YELLOW, ORANGE, RED, WHITE };

  dt_colors cb5  = { BLUE7, BLUE9 };
  dt_colors cb5f = { WHITE, WHITE };

  multilingua b0name;
  multilingua b1name;
  multilingua b2name;
  multilingua b3name;
  multilingua b4name;
  multilingua b5name;

  b0name.set_text("Button" DT_COLOR_1 " 0", ENGLISH);
  b0name.set_text("按钮" DT_COLOR_1 " 0", CHINESE_SIMPLE);
  b0name.set_text("按鈕" DT_COLOR_1 " 0", CHINESE);
  b0name.set_text("बटन" DT_COLOR_1 " 0", HINDI);
  b0name.set_text("Botón" DT_COLOR_1 " 0", SPANISH);
  b0name.set_text("Bouton" DT_COLOR_1 " 0", FRENCH);
  b0name.set_text("زر" DT_COLOR_1 " 0", ARABIC);
  b0name.set_text("বোতাম" DT_COLOR_1 " 0", BENGALI);
  b0name.set_text("Кнопка" DT_COLOR_1 " 0", UKRANIAN);
  b0name.set_text("Botão" DT_COLOR_1 " 0", PORTUGUESE);
  b0name.set_text("بٹن" DT_COLOR_1 " 0", URDU);
  b0name.set_text("Tombol" DT_COLOR_1 " 0", INDONESIAN);
  b0name.set_text("Taste" DT_COLOR_1 " 0", GERMAN);
  b0name.set_text("ボタン" DT_COLOR_1 " 0", JAPANESE);
  b0name.set_text("Κουμπί" DT_COLOR_1 " 0", GREEK);
  b0name.set_text("Pulsante" DT_COLOR_1 " 0", ITALIAN);
  b0name.set_text("Кнопка" DT_COLOR_1 " 0", RUSSIAN);
  b0name.set_text("לַחְצָן" DT_COLOR_1 " 0", HEBREW);


  
  b1name.set_text("B" DT_COLOR_1 "u" DT_COLOR_2 "t" DT_COLOR_3 "t" DT_COLOR_4 "o" DT_COLOR_5 "n" DT_COLOR_6 " 1", ENGLISH);
  b2name.set_text("Button 2", ENGLISH);
  b3name.set_text("Button 3", ENGLISH);
  b4name.set_text("Button 4", ENGLISH);
  b5name.set_text("Button " DT_COLOR_1 "5", ENGLISH);
  
  b[0] = new button(b0name,
		    5,       //  x_margin_percent
		    5,       //  y_margin_percent
		    20,      //  circle_width_percent
		    cb0,     //  textcolor
		    GREEN7,  //  fg
		    cb0f,    //  flash textcolor
		    GREEN2,  //  flash bg
		    button_push_function,  // Function to call when pushed
		    (void*) this,  // data for function
                    true,
                    true);

  b[1] = new button(b1name,
		    5,       //  x_margin_percent
		    5,       //  y_margin_percent
		    20,      //  circle_width_percent
		    cb1,     //  textcolor
		    GREY2,   //  fg
		    cb1f,    //  flash textcolor
		    BLUE,    //  flash bg
		    button_push_function,  // Function to call when pushed
		    (void*) this,  // data for function
                    true,
                    true);

  b[2] = new button(b2name,
		    5,       //  x_margin_percent
		    5,       //  y_margin_percent
		    20,      //  circle_width_percent
		    c,       //  textcolor
		    GREY2,   //  fg
		    cf,      //  flash textcolor
		    BLUE,    //  flash bg
		    button_push_function,  // Function to call when pushed
		    (void*) this,  // data for function
                    true,
                    true);

  b[3] = new button(b3name,
		    5,       //  x_margin_percent
		    5,       //  y_margin_percent
		    20,      //  circle_width_percent
		    c,       //  textcolor
		    GREY2,   //  fg
		    cf,      //  flash textcolor
		    BLUE,    //  flash bg
		    button_push_function,  // Function to call when pushed
		    (void*) this,  // data for function
                    true,
                    true);

  b[4] = new button(b4name,
		    5,       //  x_margin_percent
		    5,       //  y_margin_percent
		    20,      //  circle_width_percent
		    c,       //  textcolor
		    GREY2,   //  fg
		    cf,      //  flash textcolor
		    BLUE,    //  flash bg
		    button_push_function,  // Function to call when pushed
		    (void*) this,  // data for function
                    true,
                    true);

  b[5] = new button(b5name,
		    5,       //  x_margin_percent
		    5,       //  y_margin_percent
		    20,      //  circle_width_percent
		    cb5,     //  textcolor
		    GREY2,   //  fg
		    cb5f,    //  flash textcolor
		    GREY2,   //  flash bg
		    button_push_function,  // Function to call when pushed
		    (void*) this,  // data for function
                    true,
                    true);
  
  for(int i=0; i<NUM_BUTTONS; i++)
    if(b[i])
      add(b[i]);
}

void button_demo_window::layout()
{
  b[0]->resize(10*width/100, 10*height/100, 20*width/100, 20*height/100);
  b[1]->resize(40*width/100, 15*height/100, 40*width/100,  5*height/100);
  b[2]->resize(90*width/100, 90*height/100,  5*width/100,  5*height/100);
  b[3]->resize(70*width/100, 35*height/100,  6*width/100, 10*height/100);
  b[4]->resize(20*width/100, 85*height/100,  6*width/100, 10*height/100);
  b[5]->resize(45*width/100, 65*height/100, 10*width/100, 20*height/100);

  for(int i=0; i<6; i++)
    b[i]->layout();
  
  layout_dirty = false;
  dirty = true;
}

void button_demo_window::my_draw_box(color c, int x, int y, int w, int h)
{
  point p0(x,         y);
  point p1(x+w-1,     y);
  point p2(x+w-1, y+h-1);
  point p3(x,     y+h-1);

  draw_line_vertical   (p1, p2, c);
  draw_line_vertical   (p0, p3, c);
  draw_line_horizontal (p0, p1, c);
  draw_line_horizontal (p2, p3, c);
}

void button_demo_window::draw_dynamic()
{
  static int phase = 0;
  //clear(GREY0);

  if(phase>200)
    {
      float frac = (phase<300) ? (phase-200)/100.0 : (400-phase)/100.0;
      color c = scale(GREY7, bgcolor, frac);

      my_draw_box(c, 10*width/100, 10*height/100, 20*width/100, 20*height/100);
      my_draw_box(c, 40*width/100, 15*height/100, 40*width/100,  5*height/100);
      my_draw_box(c, 90*width/100, 90*height/100,  5*width/100,  5*height/100);
      my_draw_box(c, 70*width/100, 35*height/100,  6*width/100, 10*height/100);
      my_draw_box(c, 20*width/100, 85*height/100,  6*width/100, 10*height/100);
      my_draw_box(c, 45*width/100, 65*height/100, 10*width/100, 20*height/100);
    }

  phase=(phase+1)%400;

  if(last_text_count)
    {
      last_text_count--;

      dt_colors c;
      for(int i=0; i<MAX_COLORS; i++)
        c.c[i] = scale(last_text_colors.c[i], BLACK, last_text_count / (float) text_display_interval);
      
      set_text_size(height/25);
      draw_multicolored_text(last_text,
                             0,
                             c,
                             width/2,
                             height*19/20,
                             DRAW_TEXT_X_CENTER|DRAW_TEXT_Y_BOTTOM|DRAW_TEXT_ROTATE_0);
    }
  

  multiwindow::draw_dynamic();
}

void button_demo_window::draw_dirty()
{
  multiwindow::draw_dirty();
    
  point p0(0,0);
  point p1(width-1, 0);
  point p2(width-1, height-1);
  point p3(0, height-1);

  draw_line_vertical(p1, p2, GREY7);
  draw_line_vertical(p0, p3, GREY7);
  draw_line_horizontal(p0, p1, GREY7);
  draw_line_horizontal(p2, p3, GREY7);
  
  set_text_size(height/25);

  static bool init = true;

  static multilingua button_demo_window;
  if(init)
    {
      button_demo_window.set_text("Button Demo", ENGLISH);
      button_demo_window.set_text(U"Button 演示", CHINESE_SIMPLE);
      button_demo_window.set_text(U"Button 示範", CHINESE);
      button_demo_window.set_text(U"Button डेमो", HINDI);
      button_demo_window.set_text(U"Button manifestación", SPANISH);
      button_demo_window.set_text(U"Button démo", FRENCH);
      button_demo_window.set_text(U"Button عرض توضيحي", ARABIC);
      button_demo_window.set_text(U"Button ডেমো", BENGALI);
      button_demo_window.set_text(U"Button демоверсія", UKRANIAN);
      button_demo_window.set_text(U"Button demonstração", PORTUGUESE);
      button_demo_window.set_text(U"Button ڈیمو", URDU);
      button_demo_window.set_text(U"Button demo", INDONESIAN);
      button_demo_window.set_text(U"Button Demo", GERMAN);
      button_demo_window.set_text(U"Button デモ", JAPANESE);
      button_demo_window.set_text(U"Button διαδήλωση", GREEK);
      button_demo_window.set_text(U"Button dimostrazione", ITALIAN);
      button_demo_window.set_text(U"Button демо", RUSSIAN);
      button_demo_window.set_text(U"Button הַדגָמָה", HEBREW);
    }
  
  draw_text(button_demo_window,
	    WHITE,
	    width/2,
	    height*1/20,
	    DRAW_TEXT_X_CENTER|DRAW_TEXT_Y_TOP|DRAW_TEXT_ROTATE_0);

  set_text_size(height/50);

  color c = GREY5;
  static multilingua txt1;
  if(init)
    {
      txt1.set_text("Several buttons are scattered around this window.", ENGLISH);
      txt1.set_text(U"这个窗口周围散布着几个按钮。", CHINESE_SIMPLE);
      txt1.set_text(U"視窗周圍散佈著幾個按鈕。", CHINESE);
      txt1.set_text(U"इस विंडो में कई बटन बिखरे हुए हैं।", HINDI);
      txt1.set_text(U"Hay varios botones repartidos por esta ventana.", SPANISH);
      txt1.set_text(U"Plusieurs boutons sont dispersés dans cette fenêtre.", FRENCH);
      txt1.set_text(U"تنتشر عدة أزرار في أنحاء هذه النافذة.", ARABIC);
      txt1.set_text(U"এই উইন্ডোটিতে বেশ কয়েকটি বোতাম ছড়িয়ে ছিটিয়ে আছে।", BENGALI);
      txt1.set_text(U"Навколо цього вікна розкидано кілька кнопок.", UKRANIAN);
      txt1.set_text(U"Vários botões estão espalhados por esta janela.", PORTUGUESE);
      txt1.set_text(U"اس ونڈو کے ارد گرد کئی بٹن بکھرے ہوئے ہیں۔", URDU);
      txt1.set_text(U"Beberapa tombol tersebar di sekitar jendela ini.", INDONESIAN);
      txt1.set_text(U"In diesem Fenster sind mehrere Schaltflächen verstreut.", GERMAN);
      txt1.set_text(U"このウィンドウにはいくつかのボタンが散りばめられています。", JAPANESE);
      txt1.set_text(U"Πολλά κουμπιά είναι διάσπαρτα σε αυτό το παράθυρο.", GREEK);
      txt1.set_text(U"Diversi pulsanti sono sparsi in questa finestra.", ITALIAN);
      txt1.set_text(U"В этом окне разбросано несколько кнопок.", RUSSIAN);
      txt1.set_text(U"מספר כפתורים פזורים ברחבי חלון זה.", HEBREW);
    }
  
  draw_text(txt1,
	    c,
	    width/2,
	    height*11/40,
	    DRAW_TEXT_X_CENTER|DRAW_TEXT_Y_TOP|DRAW_TEXT_ROTATE_0);


  static multilingua txt1a;
  if(init)
    {
      txt1a.set_text("Buttons fit themselves into the specified space.", ENGLISH);
      txt1a.set_text(U"按钮会自动调整大小以适应指定的空间。", CHINESE_SIMPLE);
      txt1a.set_text(U"按鈕會自動嵌入指定位置。", CHINESE);
      txt1a.set_text(U"बटन अपने आप दी गई जगह में फिट हो जाते हैं।", HINDI);
      txt1a.set_text(U"Los botones se ajustan automáticamente al espacio especificado.", SPANISH);
      txt1a.set_text(U"Les boutons s'adaptent automatiquement à l'espace spécifié.", FRENCH);
      txt1a.set_text(U"تتكيف الأزرار تلقائيًا لتناسب المساحة المحددة.", ARABIC);
      txt1a.set_text(U"বোতামগুলো নির্দিষ্ট স্থানে নিজেদের মানিয়ে নেয়।", BENGALI);
      txt1a.set_text(U"Кнопки самі вписуються у заданий простір.", UKRANIAN);
      txt1a.set_text(U"Os botões encaixam automaticamente no espaço especificado.", PORTUGUESE);
      txt1a.set_text(U"بٹن خود کو مخصوص جگہ میں فٹ کر لیتے ہیں۔", URDU);
      txt1a.set_text(U"Tombol-tombol tersebut menyesuaikan diri dengan ruang yang telah ditentukan.", INDONESIAN);
      txt1a.set_text(U"Die Schaltflächen passen sich dem vorgegebenen Bereich an.", GERMAN);
      txt1a.set_text(U"ボタンは指定されたスペースに自動的に収まります。", JAPANESE);
      txt1a.set_text(U"Τα κουμπιά ταιριάζουν στον καθορισμένο χώρο.", GREEK);
      txt1a.set_text(U"I pulsanti si adattano automaticamente allo spazio specificato.", ITALIAN);
      txt1a.set_text(U"Кнопки автоматически подстраиваются под заданное пространство.", RUSSIAN);
      txt1a.set_text(U"הכפתורים מתאימים את עצמם למרחב שצוין.", HEBREW);
    }
  
  draw_text(txt1a,
	    c,
	    width/2,
	    height*13/40,
	    DRAW_TEXT_X_CENTER|DRAW_TEXT_Y_TOP|DRAW_TEXT_ROTATE_0);

  
  static multilingua txt2;
  if(init)
    {
      txt2.set_text("Buttons can be pushed to cause actions.", ENGLISH);
      txt2.set_text(U"按下按钮可以触发相应的动作。", CHINESE_SIMPLE);
      txt2.set_text(U"按下按鈕即可觸發動作。", CHINESE);
      txt2.set_text(U"एक्शन करवाने के लिए बटन दबाए जा सकते हैं।", HINDI);
      txt2.set_text(U"Se pueden pulsar botones para desencadenar acciones.", SPANISH);
      txt2.set_text(U"On peut appuyer sur des boutons pour déclencher des actions.", FRENCH);
      txt2.set_text(U"يمكن الضغط على الأزرار لتنفيذ إجراءات معينة.", ARABIC);
      txt2.set_text(U"বোতাম চাপলে বিভিন্ন কাজ সম্পন্ন করা যায়।", BENGALI);
      txt2.set_text(U"Кнопки можна натискати, щоб викликати дії.", UKRANIAN);
      txt2.set_text(U"Os botões podem ser premidos para executar ações.", PORTUGUESE);
      txt2.set_text(U"بٹنوں کو کارروائیوں کا سبب بننے کے لیے دھکیلا جا سکتا ہے۔", URDU);
      txt2.set_text(U"Tombol-tombol dapat ditekan untuk memicu suatu tindakan.", INDONESIAN);
      txt2.set_text(U"Durch Drücken der Tasten können Aktionen ausgelöst werden.", GERMAN);
      txt2.set_text(U"ボタンを押すことで様々な動作を実行できます。", JAPANESE);
      txt2.set_text(U"Τα κουμπιά μπορούν να πατηθούν για να προκαλέσουν ενέργειες.", GREEK);
      txt2.set_text(U"I pulsanti possono essere premuti per attivare determinate azioni.", ITALIAN);
      txt2.set_text(U"Нажатие кнопок может приводить к выполнению определенных действий.", RUSSIAN);
      txt2.set_text(U"ניתן ללחוץ על כפתורים כדי לגרום לפעולות.", HEBREW);
    }
  draw_text(txt2,
	    c,
	    width/2,
	    height*40/80,
	    DRAW_TEXT_X_CENTER|DRAW_TEXT_Y_TOP|DRAW_TEXT_ROTATE_0);

  static multilingua txt3;
  if(init)
    {
      txt3.set_text("Buttons do not stay pushed.", ENGLISH);
      txt3.set_text(U"按钮无法保持按下状态。", CHINESE_SIMPLE);
      txt3.set_text(U"按鈕不會一直保持按下狀態。", CHINESE);
      txt3.set_text(U"बटन दबे नहीं रहते।", HINDI);
      txt3.set_text(U"Los botones no se quedan pulsados.", SPANISH);
      txt3.set_text(U"Les boutons ne restent pas enfoncés.", FRENCH);
      txt3.set_text(U"الأزرار لا تبقى مضغوطة.", ARABIC);
      txt3.set_text(U"বোতামগুলো চাপা অবস্থায় থাকে না।", BENGALI);
      txt3.set_text(U"Кнопки не залишаються натиснутими.", UKRANIAN);
      txt3.set_text(U"Os botões não permanecem premidos.", PORTUGUESE);
      txt3.set_text(U"بٹن دھکیلے نہیں رہتے۔", URDU);
      txt3.set_text(U"Tombol-tombol tersebut tidak tetap dalam posisi tertekan.", INDONESIAN);
      txt3.set_text(U"Die Knöpfe bleiben nicht gedrückt.", GERMAN);
      txt3.set_text(U"ボタンが押された状態にならない。", JAPANESE);
      txt3.set_text(U"Τα κουμπιά δεν μένουν πατημένα.", GREEK);
      txt3.set_text(U"I pulsanti non rimangono premuti.", ITALIAN);
      txt3.set_text(U"Кнопки не остаются в нажатом положении.", RUSSIAN);
      txt3.set_text(U"הכפתורים לא נשארים לחוצים.", HEBREW);
    }
  draw_text(txt3,
	    c,
	    width/2,
	    height*45/80,
	    DRAW_TEXT_X_CENTER|DRAW_TEXT_Y_TOP|DRAW_TEXT_ROTATE_0);

  static multilingua txt4;
  if(init)
    {
      txt4.set_text("For controls that stay pushed, use a 'multiselect'.", ENGLISH);
      txt4.set_text(U"对于需要保持按下的控件，请使用“多选”功能。", CHINESE_SIMPLE);
      txt4.set_text(U"對於保持按下狀態的控件，請使用「多選」。", CHINESE);
      txt4.set_text(U"जो कंट्रोल लगातार दबे रहते हैं, उनके लिए 'मल्टीसेलेक्ट' का इस्तेमाल करें।", HINDI);
      txt4.set_text(U"Para los controles que permanecen pulsados, utilice la opción de \"selección múltiple\".", SPANISH);
      txt4.set_text(U"Pour les commandes qui restent enfoncées, utilisez une sélection multiple.", FRENCH);
      txt4.set_text(U"بالنسبة لعناصر التحكم التي تبقى مضغوطة، استخدم خيار \"التحديد المتعدد\".", ARABIC);
      txt4.set_text(U"যেসব কন্ট্রোল চাপ দিয়ে ধরে রাখতে হয়, সেগুলোর জন্য 'মাল্টিসিলেক্ট' ব্যবহার করুন।", BENGALI);
      txt4.set_text(U"Для елементів керування, які залишаються натиснутими, використовуйте «багаторазовий вибір».", UKRANIAN);
      txt4.set_text(U"Para os controlos que permanecem premidos, utilize uma 'seleção múltipla'.", PORTUGUESE);
      txt4.set_text(U"ایسے کنٹرولز کے لیے جو دھکیلتے رہتے ہیں، 'ملٹی سلیکٹ' استعمال کریں۔", URDU);
      txt4.set_text(U"Untuk kontrol yang tetap dalam posisi ditekan, gunakan 'pilihan ganda'.", INDONESIAN);
      txt4.set_text(U"Für Bedienelemente, die gedrückt bleiben sollen, verwenden Sie eine „Mehrfachauswahl“.", GERMAN);
      txt4.set_text(U"押しっぱなしにするコントロールには、「複数選択」機能を使用してください。", JAPANESE);
      txt4.set_text(U"Για στοιχεία ελέγχου που παραμένουν ενεργά, χρησιμοποιήστε μια «πολλαπλή επιλογή».", GREEK);
      txt4.set_text(U"Per i controlli che rimangono premuti, utilizzare la funzione di \"selezione multipla\".", ITALIAN);
      txt4.set_text(U"Для элементов управления, которые остаются нажатыми, используйте режим множественного выбора.", RUSSIAN);
      txt4.set_text(U"עבור פקדים שנשארים לחוץ, השתמשו ב'בחירה מרובה'.", HEBREW);
    }
  draw_text(txt4,
	    c,
	    width/2,
	    height*50/80,
	    DRAW_TEXT_X_CENTER|DRAW_TEXT_Y_TOP|DRAW_TEXT_ROTATE_0);

  init = false;
}


void button_push_function(button*b, void* v)
{
  button_demo_window* bdw = (button_demo_window*) v;

  static bool init = true;
  static multilingua end_text;

  if(init)
    {
      end_text.set_text("was pushed", ENGLISH);
      end_text.set_text(U"被推了", CHINESE_SIMPLE);
      end_text.set_text(U"被推", CHINESE);
      end_text.set_text(U"धक्का दिया गया", HINDI);
      end_text.set_text(U"fue empujado", SPANISH);
      end_text.set_text(U"a été poussé", FRENCH);
      end_text.set_text(U"تم دفعه", ARABIC);
      end_text.set_text(U"ধাক্কা দেওয়া হয়েছিল", BENGALI);
      end_text.set_text(U"був штовхнутий", UKRANIAN);
      end_text.set_text(U"foi empurrado", PORTUGUESE);
      end_text.set_text(U"دھکیل دیا گیا تھا", URDU);
      end_text.set_text(U"didorong", INDONESIAN);
      end_text.set_text(U"wurde geschoben", GERMAN);
      end_text.set_text(U"押された", JAPANESE);
      end_text.set_text(U"σπρώχτηκε", GREEK);
      end_text.set_text(U"è stato spinto", ITALIAN);
      end_text.set_text(U"был оттеснен", RUSSIAN);
      end_text.set_text(U"נדחף", HEBREW);

      init = false;
    }

  bdw->last_text.set_text((const char32_t*)b->name);
  bdw->last_text.append_text(DT_COLOR_21 " ");
  bdw->last_text.append_text((const char32_t*)end_text);
  bdw->last_text_count = bdw->text_display_interval;

  bdw->last_text_colors = b->TEXT_COLORS;
  bdw->last_text_colors.c[21] = WHITE;
}
