
// SPDX-License-Identifier: CC-BY-NC-SA-4.0
//
// Copyright (C) 2026 Bit by Bit Signal Processing LLC (https://bxbsp.com)
//
// This work is placed under the "Creative Commons Attribution
// NonCommercial ShareAlike 4.0 International" license, known
// by the shortened acronym "CC-BY-NC-SA-4.0".
//
// This work is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//
// A CC-BY-NC-SA-4.0 license allows you to use this work for
// noncommercial purposes so long as attribution is made to the
// original author.  Modified versions of this work may be distributed,
// but only under the same license.  For further details, see the
// Creative Commons License "CC-BY-NC-SA-4.0".
//
// You should have received a copy of the CC-BY-NC-SA-4.0 license
// along with this work. If not, see
// <https://creativecommons.org/licenses/by-nc-sa/4.0/>.
//

#include "kiosk/window.hh"
#include "kiosk/multiwindow.hh"
#include "kiosk/button.hh"
#include "kiosk/number_box.hh"
#include "kiosk/unalingua.hh"

#include "number_box_demo.hh"
#include "languages.hh"

void button_push_function_questionable_action(button*b, void* v);
void button_push_function_favorite_color(button*b, void* v);

class number_box_demo_window : public multiwindow
{
  number_box* nb1;
  number_box* nb2;
  number_box* nb3;
  number_box* nb4;
  double freq_value;
  double temp_value;
  double value;
  double positive_value;

  virtual void layout();
  virtual void draw_dirty();
  virtual void draw_dynamic();

public:

  const static int text_display_interval    = 100;

  unalingua        last_text;
  int              last_text_count;

  number_box_demo_window();
  ~number_box_demo_window();
};

window* create_number_box_demo_window()
{
  return new number_box_demo_window();
}


number_box_demo_window::~number_box_demo_window()
{
  delete nb1;
  delete nb2;
  delete nb3;
  delete nb4;
}

number_box_demo_window::number_box_demo_window()
{
  last_text_count = 0;

  static number_box_unit nbu[2] = { { "MHz", 1.0, 0.0 }, { "kHz", 1000.0, 0.0 } };

  freq_value = 3.1415927;

  multilingua min_text;
  min_text.set_text("Min ", ENGLISH);

  multilingua max_text;
  max_text.set_text(", Max ", ENGLISH);

  multilingua freqname;
  freqname.set_text("Frequency", ENGLISH);  
  freqname.set_text(U"频率", CHINESE_SIMPLE);
  freqname.set_text(U"頻率", CHINESE);
  freqname.set_text(U"आवृत्ति", HINDI);
  freqname.set_text(U"Frecuencia", SPANISH);
  freqname.set_text(U"Fréquence", FRENCH);
  freqname.set_text(U"تكرار", ARABIC);
  freqname.set_text(U"ফ্রিকোয়েন্সি", BENGALI);
  freqname.set_text(U"Частота", UKRANIAN);
  freqname.set_text(U"Frequência", PORTUGUESE);
  freqname.set_text(U"تعدد", URDU);
  freqname.set_text(U"Frekuensi", INDONESIAN);
  freqname.set_text(U"Frequenz", GERMAN);
  freqname.set_text(U"頻度", JAPANESE);
  freqname.set_text(U"Συχνότητα", GREEK);
  freqname.set_text(U"Frequenza", ITALIAN);
  freqname.set_text(U"Частота", RUSSIAN);
  freqname.set_text(U"תֶדֶר", HEBREW);

  nb1 = new number_box(freqname,
		       freq_value,   // double&          value,
		       -100.0,       // double           min_value,
		       100.0,        // double           max_value,
		       min_text,     // multilingua      min_text
		       max_text,     // multilingua      max_text
		       false,         // bool             integer_only
		       nbu,          // number_box_unit* units
		       2,            // int              num_units,
		       2,            // int              num_units_to_display,
		       WHITE,        // color            textcolor,
		       GREY5,        // color            fg,
		       BLACK);       // color            bg=BLACK);
  
  add(nb1);

  static number_box_unit nbu_temp[4] =
    {
      { U"\x00B0" U"C",     1.0,   0.0  }, // Celcius
      { U"\x00B0" U"F", 9.0/5.0,  32.0  }, // Fahrenheight
      { U"\x00B0" U"K",     1.0, 273.15 }, // Kelvin
      { U"\x00B0" U"R", 9.0/5.0, 491.67 }  // Rankine
    };

  temp_value = 50.0;

  multilingua tempname;
  tempname.set_text("Temperature", ENGLISH);  
  tempname.set_text(U"温度", CHINESE_SIMPLE);
  tempname.set_text(U"溫度", CHINESE);
  tempname.set_text(U"तापमान", HINDI);
  tempname.set_text(U"Temperatura", SPANISH);
  tempname.set_text(U"Température", FRENCH);
  tempname.set_text(U"درجة حرارة", ARABIC);
  tempname.set_text(U"তাপমাত্রা", BENGALI);
  tempname.set_text(U"Температура", UKRANIAN);
  tempname.set_text(U"Temperatura", PORTUGUESE);
  tempname.set_text(U"درجہ حرارت", URDU);
  tempname.set_text(U"Suhu", INDONESIAN);
  tempname.set_text(U"Temperatur", GERMAN);
  tempname.set_text(U"温度", JAPANESE);
  tempname.set_text(U"Θερμοκρασία", GREEK);
  tempname.set_text(U"Temperatura", ITALIAN);
  tempname.set_text(U"Температура", RUSSIAN);
  tempname.set_text(U"טֶמפֶּרָטוּרָה", HEBREW);

  nb2 = new number_box(tempname,
		       temp_value,   // double&          value,
		       -80.0,        // double           min_value,
		       100.0,        // double           max_value,
		       min_text,     // multilingua      min_text
		       max_text,     // multilingua      max_text
		       false,         // bool             integer_only
		       nbu_temp,     // number_box_unit* units
		       4,            // int              num_units,
		       2,            // int              num_units_to_display,
		       WHITE,        // color            textcolor,
		       GREY5,        // color            fg,
		       BLACK);       // color            bg=BLACK);
    

  add(nb2);

  value = 1.0;
  
  multilingua scalarname;
  scalarname.set_text("Scalar Entry", ENGLISH);  
  scalarname.set_text(U"标量", CHINESE_SIMPLE);
  scalarname.set_text(U"標量", CHINESE);
  scalarname.set_text(U"अदिश", HINDI);
  scalarname.set_text(U"Escalar", SPANISH);
  scalarname.set_text(U"Scalaire", FRENCH);
  scalarname.set_text(U"الكمية القياسية", ARABIC);
  scalarname.set_text(U"স্কেলার", BENGALI);
  scalarname.set_text(U"Скалярний", UKRANIAN);
  scalarname.set_text(U"Escalar", PORTUGUESE);
  scalarname.set_text(U"اسکیلر", URDU);
  scalarname.set_text(U"Skalar", INDONESIAN);
  scalarname.set_text(U"Skalar", GERMAN);
  scalarname.set_text(U"スカラー", JAPANESE);
  scalarname.set_text(U"Βαθμωτό", GREEK);
  scalarname.set_text(U"Scalare", ITALIAN);
  scalarname.set_text(U"Скаляр", RUSSIAN);
  scalarname.set_text(U"סקלרי", HEBREW);

  nb3 = new number_box(scalarname,
		       value,        // double&          value,
		       -10.0,        // double           min_value,
		       10.0,         // double           max_value,
		       min_text,     // multilingua      min_text
		       max_text,     // multilingua      max_text
		       false,        // bool             integer_only
		       0,            // number_box_unit* units
		       0,            // int              num_units,
		       0,            // int              num_units_to_display,
		       WHITE,        // color            textcolor,
		       GREY5,        // color            fg,
		       BLACK);       // color            bg=BLACK);
    

  add(nb3);


  positive_value = 1.0;
  
  multilingua positive_scalarname;
  positive_scalarname.set_text("Positive Integer Entry", ENGLISH);  
  positive_scalarname.set_text(U"正整数", CHINESE_SIMPLE);
  positive_scalarname.set_text(U"正整數", CHINESE);
  positive_scalarname.set_text(U"सकारात्मक पूर्णांक", HINDI);
  positive_scalarname.set_text(U"Entero positivo", SPANISH);
  positive_scalarname.set_text(U"Entier positif", FRENCH);
  positive_scalarname.set_text(U"عدد صحيح موجب", ARABIC);
  positive_scalarname.set_text(U"ধনাত্মক পূর্ণসংখ্যা", BENGALI);
  positive_scalarname.set_text(U"Додатне ціле число", UKRANIAN);
  positive_scalarname.set_text(U"Número inteiro positivo", PORTUGUESE);
  positive_scalarname.set_text(U"مثبت عدد", URDU);
  positive_scalarname.set_text(U"Bilangan bulat positif", INDONESIAN);
  positive_scalarname.set_text(U"Positive ganze Zahl", GERMAN);
  positive_scalarname.set_text(U"正の整数", JAPANESE);
  positive_scalarname.set_text(U"Θετικός ακέραιος", GREEK);
  positive_scalarname.set_text(U"Numero intero positivo", ITALIAN);
  positive_scalarname.set_text(U"Положительное целое число", RUSSIAN);
  positive_scalarname.set_text(U"מספר שלם חיובי", HEBREW);

  nb4 = new number_box(positive_scalarname,
		       positive_value,        // double&          value,
		       0.0,          // double           min_value,
		       100.0,        // double           max_value,
		       min_text,     // multilingua      min_text
		       max_text,     // multilingua      max_text
		       true,         // bool             integer_only
		       0,            // number_box_unit* units
		       0,            // int              num_units,
		       0,            // int              num_units_to_display,
		       WHITE,        // color            textcolor,
		       GREY5,        // color            fg,
		       BLACK);       // color            bg=BLACK);
    

  add(nb4);
}

void number_box_demo_window::layout()
{
  //b->resize(20*width/100, 60*height/100, 60*width/100, 20*height/100);
  nb1->resize(width/2-width/8, height/8, width/4, height/4);
  nb1->layout();

  nb2->resize(width/16, 4*height/8, width/4, height/4);
  nb2->layout();

  nb3->resize(3*width/8, 4*height/8, width/4, height/4);
  nb3->layout();

  nb4->resize(width-width/4-width/8, 4*height/8, width/4, height/4);
  nb4->layout();

  layout_dirty = false;
  dirty = true;
}

void number_box_demo_window::draw_dynamic()
{
  //clear(GREY0);

  if(last_text_count)
    {
      last_text_count--;

      color c = scale(WHITE, BLACK, last_text_count / (float) text_display_interval);
      
      set_text_size(height/25);

      //draw_line(point(0, height*60/100), point(width-1, height*60/100), RED);
      
      draw_text(last_text,
		c,
		width/2,
		height*60/100,
		DRAW_TEXT_X_CENTER|DRAW_TEXT_Y_TOP|DRAW_TEXT_ROTATE_0);
    }
  

  multiwindow::draw_dynamic();
}

void number_box_demo_window::draw_dirty()
{
  multiwindow::draw_dirty();
    
  point p0(0,0);
  point p1(width-1, 0);
  point p2(width-1, height-1);
  point p3(0, height-1);

  draw_line_vertical(p1, p2, GREY7);
  draw_line_vertical(p0, p3, GREY7);
  draw_line_horizontal(p0, p1, GREY7);
  draw_line_horizontal(p2, p3, GREY7);
  
  set_text_size(height/25);

  static bool init = true;

  static multilingua number_box_demo;
  if(init)
    {
      number_box_demo.set_text("Number_Box Demo", ENGLISH);
      number_box_demo.set_text(U"Number_Box 演示", CHINESE_SIMPLE);
      number_box_demo.set_text(U"Number_Box 示範", CHINESE);
      number_box_demo.set_text(U"Number_Box डेमो", HINDI);
      number_box_demo.set_text(U"Number_Box manifestación", SPANISH);
      number_box_demo.set_text(U"Number_Box démo", FRENCH);
      number_box_demo.set_text(U"Number_Box عرض توضيحي", ARABIC);
      number_box_demo.set_text(U"Number_Box ডেমো", BENGALI);
      number_box_demo.set_text(U"Number_Box демоверсія", UKRANIAN);
      number_box_demo.set_text(U"Number_Box demonstração", PORTUGUESE);
      number_box_demo.set_text(U"Number_Box ڈیمو", URDU);
      number_box_demo.set_text(U"Number_Box demo", INDONESIAN);
      number_box_demo.set_text(U"Number_Box Demo", GERMAN);
      number_box_demo.set_text(U"Number_Box デモ", JAPANESE);
      number_box_demo.set_text(U"Number_Box διαδήλωση", GREEK);
      number_box_demo.set_text(U"Number_Box dimostrazione", ITALIAN);
      number_box_demo.set_text(U"Number_Box демо", RUSSIAN);
      number_box_demo.set_text(U"Number_Box הַדגָמָה", HEBREW);
    }
  
  draw_text(number_box_demo,
	    WHITE,
	    width/2,
	    height*1/20,
	    DRAW_TEXT_X_CENTER|DRAW_TEXT_Y_TOP|DRAW_TEXT_ROTATE_0);

  set_text_size(height/50);

  color c = GREY5;
  static multilingua txt1;
  if(init)
    {
      txt1.set_text("Numberic Entry can be made with a 'number_box'.", ENGLISH);
      txt1.set_text(U"可以使用“数字输入框”进行数字输入。", CHINESE_SIMPLE);
      txt1.set_text(U"可以使用“數字框”輸入數字。", CHINESE);
      txt1.set_text(U"न्यूमेरिक एंट्री 'नंबर बॉक्स' से की जा सकती है।", HINDI);
      txt1.set_text(U"La introducción de datos numéricos se puede realizar mediante un \"cuadro numérico\".", SPANISH);
      txt1.set_text(U"La saisie numérique peut être effectuée à l'aide d'une « zone de saisie numérique ».", FRENCH);
      txt1.set_text(U"يمكن إدخال القيم الرقمية باستخدام \"مربع الأرقام\".", ARABIC);
      txt1.set_text(U"সংখ্যাসূচক এন্ট্রি একটি 'নাম্বার বক্স'-এর মাধ্যমে করা যেতে পারে।", BENGALI);
      txt1.set_text(U"Числовий ввід можна здійснити за допомогою «числового поля».", UKRANIAN);
      txt1.set_text(U"A introdução de números pode ser feita com uma 'caixa de números'.", PORTUGUESE);
      txt1.set_text(U"عددی اندراج ایک 'نمبر باکس' کے ساتھ کیا جا سکتا ہے۔", URDU);
      txt1.set_text(U"Input numerik dapat dilakukan dengan menggunakan 'kotak angka'.", INDONESIAN);
      txt1.set_text(U"Die Eingabe von Zahlen kann über ein „Zahlenfeld“ erfolgen.", GERMAN);
      txt1.set_text(U"数値入力は「数値入力ボックス」を使用して行うことができます。", JAPANESE);
      txt1.set_text(U"Η αριθμητική εισαγωγή μπορεί να γίνει με ένα «αριθμητικό πλαίσιο».", GREEK);
      txt1.set_text(U"L'inserimento di valori numerici può essere effettuato tramite una \"casella numerica\".", ITALIAN);
      txt1.set_text(U"Ввод числовых значений можно осуществлять с помощью поля для ввода чисел.", RUSSIAN);
      txt1.set_text(U"ניתן לבצע הזנה מספרית באמצעות 'תיבת מספרים'.", HEBREW);
    }
  
  draw_text(txt1,
	    c,
	    width/2,
	    height*11/40,
	    DRAW_TEXT_X_CENTER|DRAW_TEXT_Y_TOP|DRAW_TEXT_ROTATE_0);


  static multilingua txt1a;
  if(init)
    {
      txt1a.set_text("The number_box allows entry in multiple units.", ENGLISH);
      txt1a.set_text(U"该数字输入框允许输入多种单位的数值。", CHINESE_SIMPLE);
      txt1a.set_text(U"數字框允許輸入多個單位。", CHINESE);
      txt1a.set_text(U"नंबर_बॉक्स कई यूनिट में एंट्री की अनुमति देता है।", HINDI);
      txt1a.set_text(U"El cuadro numérico permite introducir valores en múltiples unidades.", SPANISH);
      txt1a.set_text(U"La zone de saisie numérique permet la saisie de valeurs dans plusieurs unités.", FRENCH);
      txt1a.set_text(U"يسمح مربع الإدخال الرقمي بإدخال القيم بوحدات متعددة.", ARABIC);
      txt1a.set_text(U"নাম্বার_বক্সটি একাধিক এককে ডেটা প্রবেশের সুযোগ দেয়।", BENGALI);
      txt1a.set_text(U"Поле number_box дозволяє введення в кількох одиницях.", UKRANIAN);
      txt1a.set_text(U"A caixa de números permite a entrada de múltiplas unidades.", PORTUGUESE);
      txt1a.set_text(U"number_box متعدد اکائیوں میں داخلے کی اجازت دیتا ہے۔", URDU);
      txt1a.set_text(U"Kotak angka memungkinkan penginputan dalam berbagai satuan.", INDONESIAN);
      txt1a.set_text(U"Das Zahlenfeld ermöglicht die Eingabe in verschiedenen Einheiten.", GERMAN);
      txt1a.set_text(U"この数値入力ボックスでは、複数の単位での入力が可能です。", JAPANESE);
      txt1a.set_text(U"Το number_box επιτρέπει την εισαγωγή σε πολλαπλές μονάδες.", GREEK);
      txt1a.set_text(U"Il campo numerico consente l'inserimento di valori in diverse unità di misura.", ITALIAN);
      txt1a.set_text(U"Поле ввода числа позволяет вводить значения в нескольких единицах измерения.", RUSSIAN);
      txt1a.set_text(U"תיבת המספרים מאפשרת הזנה במספר יחידות.", HEBREW);
    }
  
  draw_text(txt1a,
	    c,
	    width/2,
	    height*13/40,
	    DRAW_TEXT_X_CENTER|DRAW_TEXT_Y_TOP|DRAW_TEXT_ROTATE_0);

  
  static multilingua txt2;
  if(init)
    {
      txt2.set_text("Number boxes without units are also available.", ENGLISH);
      txt2.set_text(U"不带单位的数字框也可用。", CHINESE_SIMPLE);
      txt2.set_text(U"也提供不含單位的數字箱。", CHINESE);
      txt2.set_text(U"बिना यूनिट वाले नंबर बॉक्स भी उपलब्ध हैं।", HINDI);
      txt2.set_text(U"También hay disponibles casillas numéricas sin unidades.", SPANISH);
      txt2.set_text(U"Des boîtes contenant des chiffres sans unités sont également disponibles.", FRENCH);
      txt2.set_text(U"تتوفر أيضًا صناديق أرقام بدون وحدات قياس.", ARABIC);
      txt2.set_text(U"এককবিহীন সংখ্যার বাক্সও পাওয়া যায়।", BENGALI);
      txt2.set_text(U"Також доступні числові ящики без одиниць вимірювання.", UKRANIAN);
      txt2.set_text(U"Caixas numéricas sem unidades também estão disponíveis.", PORTUGUESE);
      txt2.set_text(U"یونٹس کے بغیر نمبر بکس بھی دستیاب ہیں۔", URDU);
      txt2.set_text(U"Kotak angka tanpa satuan juga tersedia.", INDONESIAN);
      txt2.set_text(U"Es sind auch Zahlenfelder ohne Einheiten erhältlich.", GERMAN);
      txt2.set_text(U"単位のない数字入力欄もご利用いただけます。", JAPANESE);
      txt2.set_text(U"Διατίθενται επίσης κουτιά αριθμών χωρίς μονάδες.", GREEK);
      txt2.set_text(U"Sono disponibili anche caselle numeriche senza unità di misura.", ITALIAN);
      txt2.set_text(U"Также доступны числовые поля без указания единиц измерения.", RUSSIAN);
      txt2.set_text(U"קיימות גם תיבות מספרים ללא יחידות.", HEBREW);
    }
  draw_text(txt2,
	    c,
	    width/2,
	    height*55/80,
	    DRAW_TEXT_X_CENTER|DRAW_TEXT_Y_TOP|DRAW_TEXT_ROTATE_0);

  static multilingua txt3;
  if(init)
    {
      txt3.set_text("Range-checking is performed before entries are allowed.", ENGLISH);
      txt3.set_text(U"在允许输入之前会进行范围检查。", CHINESE_SIMPLE);
      txt3.set_text(U"在允許輸入之前，會進行範圍檢查。", CHINESE);
      txt3.set_text(U"एंट्री की अनुमति देने से पहले रेंज-चेकिंग की जाती है।", HINDI);
      txt3.set_text(U"La comprobación del rango de valores se realiza antes de permitir la introducción de datos.", SPANISH);
      txt3.set_text(U"Une vérification des limites est effectuée avant que les entrées ne soient autorisées.", FRENCH);
      txt3.set_text(U"يتم إجراء التحقق من النطاق قبل السماح بإدخال البيانات.", ARABIC);
      txt3.set_text(U"এন্ট্রি অনুমোদনের আগে পরিসীমা যাচাই করা হয়।", BENGALI);
      txt3.set_text(U"Перевірка діапазону виконується перед тим, як дозволити записи.", UKRANIAN);
      txt3.set_text(U"A verificação de intervalo é realizada antes de as entradas serem permitidas.", PORTUGUESE);
      txt3.set_text(U"اندراجات کی اجازت سے پہلے رینج چیکنگ کی جاتی ہے۔", URDU);
      txt3.set_text(U"Pemeriksaan rentang nilai dilakukan sebelum entri diizinkan.", INDONESIAN);
      txt3.set_text(U"Vor der Eingabe der Werte wird eine Bereichsprüfung durchgeführt.", GERMAN);
      txt3.set_text(U"値の範囲チェックは、入力が許可される前に行われます。", JAPANESE);
      txt3.set_text(U"Ο έλεγχος εύρους ζώνης πραγματοποιείται πριν επιτραπούν οι εισαγωγές.", GREEK);
      txt3.set_text(U"Il controllo dei limiti di validità viene eseguito prima che i dati vengano accettati.", ITALIAN);
      txt3.set_text(U"Проверка допустимого диапазона значений выполняется до того, как данные будут приняты.", RUSSIAN);
      txt3.set_text(U"בדיקת טווח מתבצעת לפני שמותר כניסה.", HEBREW);
    }
  draw_text(txt3,
	    c,
	    width/2,
	    height*60/80,
	    DRAW_TEXT_X_CENTER|DRAW_TEXT_Y_TOP|DRAW_TEXT_ROTATE_0);

  static multilingua txt4;
  if(init)
    {
      txt4.set_text("If the range is all positive, the '+/-' disappears.", ENGLISH);
      txt4.set_text(U"如果范围内的数值全部为正数，则“+/-”符号会消失。", CHINESE_SIMPLE);
      txt4.set_text(U"如果範圍全部為正數，則「+/-」符號將消失。", CHINESE);
      txt4.set_text(U"अगर रेंज पूरी तरह पॉजिटिव है, तो '+/-' गायब हो जाता है।", HINDI);
      txt4.set_text(U"Si el rango es completamente positivo, el signo '+/-' desaparece.", SPANISH);
      txt4.set_text(U"Si toutes les valeurs de la plage sont positives, le signe « +/- » disparaît.", FRENCH);
      txt4.set_text(U"إذا كانت جميع القيم ضمن النطاق موجبة، تختفي علامة \"+/-\".", ARABIC);
      txt4.set_text(U"যদি পরিসরটি সম্পূর্ণ ধনাত্মক হয়, তাহলে ‘+/-’ চিহ্নটি অদৃশ্য হয়ে যায়।", BENGALI);
      txt4.set_text(U"Якщо весь діапазон позитивний, позначка «+/-» зникає.", UKRANIAN);
      txt4.set_text(U"Se o intervalo for todo positivo, o '+/-' desaparece.", PORTUGUESE);
      txt4.set_text(U"اگر رینج تمام مثبت ہے، تو '+/-' غائب ہو جائے گا۔", URDU);
      txt4.set_text(U"Jika rentang nilainya semuanya positif, tanda '+/-' akan hilang.", INDONESIAN);
      txt4.set_text(U"Wenn der gesamte Wertebereich positiv ist, verschwindet das „+/-“.", GERMAN);
      txt4.set_text(U"範囲内の値がすべて正の値であれば、「+/-」の記号は消えます。", JAPANESE);
      txt4.set_text(U"Εάν το εύρος είναι όλο θετικό, το '+/-' εξαφανίζεται.", GREEK);
      txt4.set_text(U"Se l'intervallo è composto interamente da valori positivi, il simbolo \"+/-\" scompare.", ITALIAN);
      txt4.set_text(U"Если весь диапазон значений положительный, знак «+/-» исчезает.", RUSSIAN);
      txt4.set_text(U"אם הטווח חיובי כולו, הסימן '+/-' נעלם.", HEBREW);
    }
  draw_text(txt4,
	    c,
	    width/2,
	    height*65/80,
	    DRAW_TEXT_X_CENTER|DRAW_TEXT_Y_TOP|DRAW_TEXT_ROTATE_0);

  static multilingua txt5;
  if(init)
    {
      txt5.set_text("For integer-only entry, the decimal point disappears.", ENGLISH);
      txt5.set_text(U"如果只输入整数，则小数点会消失。", CHINESE_SIMPLE);
      txt5.set_text(U"如果只輸入整數，則小數點消失。", CHINESE);
      txt5.set_text(U"केवल पूर्णांक एंट्री के लिए, दशमलव बिंदु गायब हो जाता है।", HINDI);
      txt5.set_text(U"Para la introducción de números enteros, el punto decimal desaparece.", SPANISH);
      txt5.set_text(U"Pour la saisie de nombres entiers uniquement, la virgule décimale disparaît.", FRENCH);
      txt5.set_text(U"عند إدخال الأرقام الصحيحة فقط، تختفي العلامة العشرية.", ARABIC);
      txt5.set_text(U"শুধুমাত্র পূর্ণসংখ্যা প্রবেশের ক্ষেত্রে দশমিক বিন্দুটি অদৃশ্য হয়ে যায়।", BENGALI);
      txt5.set_text(U"Для введення лише цілих чисел десяткова кома зникає.", UKRANIAN);
      txt5.set_text(U"Para entradas que utilizam apenas números inteiros, o ponto decimal desaparece.", PORTUGUESE);
      txt5.set_text(U"صرف عددی اندراج کے لیے، اعشاریہ نقطہ غائب ہو جاتا ہے۔", URDU);
      txt5.set_text(U"Untuk input berupa bilangan bulat saja, titik desimal akan menghilang.", INDONESIAN);
      txt5.set_text(U"Bei der Eingabe von Ganzzahlen verschwindet das Dezimalzeichen.", GERMAN);
      txt5.set_text(U"整数のみを入力する場合、小数点が表示されなくなります。", JAPANESE);
      txt5.set_text(U"Για εισαγωγή μόνο ακέραιων αριθμών, η υποδιαστολή εξαφανίζεται.", GREEK);
      txt5.set_text(U"Quando si inseriscono solo numeri interi, il punto decimale scompare.", ITALIAN);
      txt5.set_text(U"При вводе только целых чисел десятичная точка исчезает.", RUSSIAN);
      txt5.set_text(U"עבור הזנת מספר שלם בלבד, הנקודה העשרונית נעלמת.", HEBREW);
    }
  draw_text(txt5,
	    c,
	    width/2,
	    height*70/80,
	    DRAW_TEXT_X_CENTER|DRAW_TEXT_Y_TOP|DRAW_TEXT_ROTATE_0);

  init = false;
}


